# -*- coding: utf-8 -*-

import logging
import os
import shutil

from sandbox import sdk2

from sandbox.projects.common.arcadia import sdk as arc
from sandbox.projects.common.constants import constants as arcc
from sandbox.projects.common.nanny import nanny
from sandbox.common.errors import TaskFailure
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.common.types import client as ctc

from sandbox.projects.dj.process_wrapper import process_wrapper


logger = logging.getLogger("DjCompileConfig")


class DjCompileConfig(nanny.ReleaseToNannyTask2, sdk2.Task):
    """Make Compile recommender config"""

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.LINUX_PRECISE | ctc.Tag.LINUX_TRUSTY | ctc.Tag.LINUX_XENIAL

    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl("Arcadia URL", required=True)
        arcadia_revision = sdk2.parameters.String("Revision", required=False)

        config_folder_path = sdk2.parameters.String("Config folder path", required=True)
        python_config_filename = sdk2.parameters.String("Python config filename", required=True)
        generator_method = sdk2.parameters.String("Method name to generate config", required=True)
        options = sdk2.parameters.String("Additional options for config compiler", required=False)

        output_type = sdk2.parameters.String("Output type for config compiler", required=True,
                                             default_value="destination-recommender-config")

        output_resource_type = sdk2.parameters.String("Resource type for output config",
                                                      required=True,
                                                      default_value="DJ_COMPILED_CONFIG")
        output_filename = sdk2.parameters.String("Output filename", required=True,
                                                 default_value="recommender_config.pbtxt")

        config_compiler_resource = sdk2.parameters.Resource("Config compiler built resource",
                                                            required=False)

        config_compiler_path = sdk2.parameters.String("Config compiler path",
                                                      required=False,
                                                      default="dj/tools/config_compiler/config_compiler")

    def get_arcadia_url(self):
        if not self.Parameters.arcadia_revision:
            return self.Parameters.arcadia_url
        return '{}@{}'.format(self.Parameters.arcadia_url, self.Parameters.arcadia_revision)

    def build_config_compiler(self, arcadia_dir, copy_file=False):
        logger.info("Start to make config compiler")
        make_target = os.path.dirname(self.Parameters.config_compiler_path)
        arc.do_build(
            build_system=arcc.SEMI_DISTBUILD_BUILD_SYSTEM,
            source_root=arcadia_dir,
            targets=[make_target],
            results_dir=arcadia_dir,
            clear_build=False,
        )
        logger.info("End to make config compiler")
        return os.path.join(arcadia_dir, self.Parameters.config_compiler_path)

    def on_execute(self):
        logger.info("Start process")

        if not self.Parameters.config_compiler_resource and not self.Parameters.config_compiler_path:
            raise TaskFailure('You should specify config compiler binary or path to build it')

        output_resource = sdk2.Resource[self.Parameters.output_resource_type](
            self, 'Compiled config', self.Parameters.output_filename)
        output_resource_data = sdk2.ResourceData(output_resource)

        with arc.mount_arc_path(self.get_arcadia_url()) as arcadia_dir:

            if self.Parameters.config_compiler_resource:
                os.mkdir("tools")
                shutil.copyfile(str(sdk2.ResourceData(self.Parameters.config_compiler_resource).path),
                                os.path.join("tools", "config_compiler"))
                config_compiler_binary = os.path.join("tools", "config_compiler")
                sp.check_output(["chmod", "+x", config_compiler_binary])
            else:
                config_compiler_binary = self.build_config_compiler(arcadia_dir, copy_file=True)

            logger.info("Start generation")
            config_full_path = os.path.join(arcadia_dir,
                                            self.Parameters.config_folder_path,
                                            self.Parameters.python_config_filename)

            command = [
                config_compiler_binary,
                "--name",
                self.Parameters.generator_method,
                "--config",
                config_full_path,
                "--{}".format(self.Parameters.output_type),
                str(output_resource_data.path)
            ]

            additional_options = self.Parameters.options
            if additional_options:
                command.extend(["--options", additional_options])

            with process_wrapper(self, logger='config_compiler') as pl:
                build_output = sp.check_output(command, stderr=pl.stderr)

            logger.info("End generation")

        logger.info("Finished")

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)
