import base64
import os
import requests

from urlparse import urlparse

import sandbox.common.types.client as ctc

from sandbox import sdk2

from sandbox.common.types.misc import NotExists

from sandbox.projects.dj.dj_report import DjReport
from sandbox.projects.dj.server import BasesearchRecommender


class DjGetBasesearchResponses(sdk2.Task):
    stats_key = "stats"

    stats_types = (
        ("empty_quant", "Empty response quantile", "{:0.02f}"),
        ("avg_doc_count", "Average document count", "{:0.02f}"),
        ("avg_response_size", "Average response size", "{:0.02f}kb"),
        ("queries_count", "Total queries count", "{}")
    )

    class Parameters(sdk2.Task.Parameters):
        basesearch_resource = sdk2.parameters.Resource(
            'Basesearch basesearch resource',
            required=True
        )
        basesearch_config_resource = sdk2.parameters.Resource(
            'Basesearch config resource',
            required=True
        )
        resource_bundle_resource = sdk2.parameters.Resource(
            'Resource bundle resource',
            required=True
        )
        shard_resource = sdk2.parameters.Resource(
            'Shard resource',
            required=True
        )
        request_count = sdk2.parameters.Integer(
            "Count of requests",
            default=1000,
            required=True
        )
        dolbilka_plan_resource = sdk2.parameters.Resource(
            "Resource with dolbilka plan",
            required=True
        )
        responses_resource_type = sdk2.parameters.String(
            "Responses resource type",
            default="OTHER_RESOURCE",
            required=True
        )
        ram = sdk2.parameters.Integer(
            'Ram gb',
            default=10,
            required=True)
        disk_space = sdk2.parameters.Integer(
            'Disk space gb',
            default=20,
            required=True)

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC & ctc.Tag.LINUX_PRECISE & ctc.Tag.INTEL_E5_2650

    def parse_next_request(self, f):
        url = f.readline().strip()
        if url == '':
            return None
        o = urlparse(url)
        return 'http://localhost:{}/{}?{}&ms=proto&pron=dj_hr_attrs=da'.format(self.basesearch.get_port(), '/yandsearch', o.query)

    def send_request(self, url):
        for i in range(5):
            response = requests.get(url)
            if response.ok:
                return response.content
        raise RuntimeError('Error in request processing. Url:\n{}'.format(url))

    def on_save(self):
        self.Requirements.ram = self.Parameters.ram * 1024
        self.Requirements.disk_space = self.Parameters.disk_space * 1024

    def on_execute(self):
        params = self.Parameters
        os.mkdir('resources')
        os.symlink(str(sdk2.ResourceData(params.resource_bundle_resource).path),
            os.path.join(os.getcwd(), 'resources/resources'))

        queries_resource_data = sdk2.ResourceData(params.dolbilka_plan_resource)

        responses_resource = sdk2.Resource[params.responses_resource_type](
            self, 'BasesearchResponses', 'responses')
        responses_resource_data = sdk2.ResourceData(responses_resource)

        self.basesearch = BasesearchRecommender(
            basesearch_path=str(sdk2.ResourceData(params.basesearch_resource).path),
            basesearch_conf_path=str(sdk2.ResourceData(params.basesearch_config_resource).path),
            shard_path=str(sdk2.ResourceData(params.shard_resource).path)
        )

        total_zero_doc_count = 0.0
        total_doc_count = 0.0
        total_response_size = 0.0
        total_query_count = 0

        with open(str(queries_resource_data.path), 'r') as queries_file, open(str(responses_resource_data.path), 'w') as responses_file:
            for i in range(self.Parameters.request_count):
                next_request = self.parse_next_request(queries_file)
                if not next_request:
                    break

                response = self.send_request(next_request)

                dj_report = DjReport(response)

                responses_file.write(base64.b64encode(next_request))
                responses_file.write('\t')
                responses_file.write(base64.b64encode(response))
                responses_file.write('\n')

                total_query_count += 1
                total_response_size += len(response)
                doc_count = len(dj_report.docs)
                total_doc_count += doc_count
                total_zero_doc_count += 1 if doc_count == 0 else 0

        stats = {
            'avg_doc_count': total_doc_count / total_query_count,
            'avg_response_size': total_response_size / 1e3 / total_query_count,
            'empty_quant': total_zero_doc_count / total_query_count,
            'queries_count': total_query_count
        }

        if getattr(self.Context, self.stats_key) is NotExists:
            setattr(self.Context, self.stats_key, stats)

        responses_resource_data.ready()

    @sdk2.footer()
    def footer(self):
        if getattr(self.Context, self.stats_key) is NotExists:
            return "Calculating..."

        stats = getattr(self.Context, self.stats_key)

        body = ""
        for key, title, fmt in self.stats_types:
            body += "<tr><td style=\"border: 1px; border-style: solid\">{}</td><td style=\"border: 1px; border-style: solid\">{}</td></tr>".format(key, self._format_stats(fmt, stats, key))

        return "<h3>Stats</h3><table style=\"border: 1px; border-collapse: collapse; border-style: solid\">{}</table>".format(body)

    def _format_stats(self, fmt, stats, key):
        return fmt.format(stats[key])

