# -*- coding: utf-8 -*-

import fnmatch
import logging
import os
import tarfile

from sandbox import sdk2

from sandbox.projects.common.arcadia import sdk as arc
from sandbox.projects.common.constants import constants as arcc

from sandbox.projects.dj.resource_types import DjEmbeddedYqlArchive

from sandbox.common.types import client as ctc
from sandbox.common.errors import TaskFailure


logger = logging.getLogger("DjMakeEmbeddedYql")


class DjMakeEmbeddedYql(sdk2.Task):
    """Build yql_embedded archive for running YQL nirvana operations in embedded mode"""

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.LINUX_PRECISE | ctc.Tag.LINUX_TRUSTY | ctc.Tag.LINUX_XENIAL

    class Parameters(sdk2.Task.Parameters):
        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl("Arcadia URL", required=True)

        udfs = sdk2.parameters.List("UDFs", sdk2.parameters.String, required=False, default=["common"])

        thinlto = sdk2.parameters.Bool("Build with ThinLTO", required=True, default=False)

    def on_execute(self):
        targets = ["yql/cli"]
        targets_thinlto = ["yql/tools/mrjob", "yql/tools/udf_resolver"]
        artifacts = ["yql/cli/bin/yql", "yql/tools/mrjob/mrjob", "yql/tools/udf_resolver/udf_resolver"]

        if not self.Parameters.thinlto:
            targets, targets_thinlto = targets + targets_thinlto, []

        if self.Parameters.udfs:
            for udf in self.Parameters.udfs:
                if udf not in {"-", "_"}:
                    udf_directory = "yql/udfs" + ("/" + udf if udf and udf not in {".", "*"} else "")
                    targets.append(udf_directory)
                    artifacts.append(udf_directory + "/*.so")

        logger.info("Targers: %s", targets)
        logger.info("Targers thinlto: %s", targets_thinlto)
        logger.info("Artifacts: %s", artifacts)

        with arc.mount_arc_path(self.Parameters.checkout_arcadia_from_url) as arcadia_dir:
            assert arcadia_dir.find("*") < 0

            logging.info("Building projects")

            if targets:
                arc.do_build(
                    build_system=arcc.DISTBUILD_BUILD_SYSTEM,
                    source_root=arcadia_dir,
                    targets=targets,
                    results_dir=arcadia_dir,
                    clear_build=False
                )

            if targets_thinlto:
                arc.do_build(
                    build_system=arcc.DISTBUILD_BUILD_SYSTEM,
                    source_root=arcadia_dir,
                    targets=targets_thinlto,
                    results_dir=arcadia_dir,
                    clear_build=False,
                    thinlto=True
                )

            logging.info("Packing arcadia")

            with tarfile.open(str(self.path("yql_embedded.tar.gz")), "w:gz", dereference=True) as tar:
                tar_dict = {}

                def tar_add(file_path):
                    file_name = os.path.join("yql_embedded", os.path.basename(file_path))
                    if file_name not in tar_dict:
                        tar_dict[file_name] = file_path
                        tar.add(file_path, file_name)
                    elif tar_dict[file_name] != file_path:
                        raise TaskFailure("UDFs file name collision")

                for artifact in artifacts:
                    apath = os.path.join(arcadia_dir, artifact)
                    if apath.find("*") >= 0:
                        for directory, subdirs, files in os.walk(os.path.dirname(apath.split('*', 1)[0])):
                            if directory in (os.path.join(arcadia_dir, dir) for dir in self.exclude_implicit):
                                if directory != os.path.dirname(apath):
                                    continue
                            for file in files:
                                path = os.path.join(directory, file)
                                if fnmatch.fnmatch(path, apath) and os.path.isfile(path) and os.access(path, os.X_OK):
                                    tar_add(path)
                    else:
                        assert os.path.isfile(apath) and os.access(apath, os.X_OK)
                        tar_add(apath)

            logging.info("Creating resource")

            sdk2.ResourceData(DjEmbeddedYqlArchive(
                self, "Built yql_embedded archive for running embedded mode", "yql_embedded.tar.gz"
            )).ready()

            logging.info("Finish")

    exclude_implicit = ["yql/udfs/common/python/python3_small", "yql/udfs/common/python/python_arc_small"]
