# -*- coding: utf-8 -*-

import hashlib
import logging
import os
import pipes
import shutil
import tempfile

from sandbox import sdk2

from sandbox.projects.common.arcadia import sdk as arc
from sandbox.projects.common.constants import constants as arcc
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.common.types import client as ctc

from sandbox.projects.dj.process_wrapper import process_wrapper


logger = logging.getLogger("DjNirvanaMake")


class DjNirvanaMake(sdk2.Task):
    """Make Nirvana operations from Arcadia"""

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.LINUX_PRECISE | ctc.Tag.LINUX_TRUSTY | ctc.Tag.LINUX_XENIAL

    class Parameters(sdk2.Task.Parameters):

        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl("Arcadia URL", required=True)

        with sdk2.parameters.RadioGroup("Mode", required=True) as mode:
            mode.values.update = mode.Value(default=True)
            mode.values.debug = None
            mode.values.create = None

        operation_path = sdk2.parameters.String("Operation path", required=True)

        root_path = sdk2.parameters.String("Root",
                                           required=True,
                                           default="yt://hahn/home/mlmarines/common/nirvana_make/operations")

        import_path = sdk2.parameters.String("Import",
                                             required=True,
                                             default="dj/nirvana/operations")

        nirvana_token = sdk2.parameters.Vault("Nirvana token",
                                              required=True,
                                              default="sec-01dbj4vf7686sazq9qe8fjqvvq[nirvana]")  # robot-mlmarines

        yt_token = sdk2.parameters.Vault("YT token",
                                         required=True,
                                         default="sec-01dbj4vf7686sazq9qe8fjqvvq[yt]")  # robot-mlmarines

        sandbox_token = sdk2.parameters.Vault("Sandbox token",
                                              required=False,
                                              default="sec-01dbj4vf7686sazq9qe8fjqvvq[sandbox]")  # robot-mlmarines

        nirvana_make_path = sdk2.parameters.String("NirvanaMake path",
                                                   required=True,
                                                   default="dj/nirvana/nirvana_make/bin/nirvana_make")

        nirvana_make_arcadia_url = sdk2.parameters.ArcadiaUrl("NirvanaMake arcadia URL", required=False, default="")

    def build_nirvana_make(self, arcadia_dir, copy_file=False):
        logger.info("Building nirvana_make: %s", self.Parameters.nirvana_make_path)
        nirvana_make_binary = self.Parameters.nirvana_make_path
        nirvana_make_target = os.path.dirname(nirvana_make_binary)
        arc.do_build(
            build_system=arcc.DISTBUILD_BUILD_SYSTEM,
            source_root=arcadia_dir,
            targets=[nirvana_make_target],
            results_dir=arcadia_dir,
            clear_build=False,
        )
        if copy_file:
            shutil.copy(os.path.join(arcadia_dir, nirvana_make_binary), self.path("nirvana_make"))
            return self.path("nirvana_make")
        else:
            return os.path.join(arcadia_dir, nirvana_make_binary)

    def build_operations(self, arcadia_dir, build_output):
        build_command_set = set()
        fake_svn_root = tempfile.mkdtemp()
        try:
            os.symlink(arcadia_dir, os.path.join(fake_svn_root, 'arcadia'))
            for build_command in build_output.split("\n"):
                build_command = build_command.strip()
                if build_command in build_command_set or not build_command:
                    continue
                build_command_set.add(build_command)
                command, full_path = build_command.split(" ", 1) if build_command.find(" ") > 0 else (build_command, "")
                if command not in {"make", "package"}:
                    raise Exception("Invalid build command: {}".format(build_command))
                if full_path.startswith(os.path.join(arcadia_dir, "")):
                    build_path = full_path[len(os.path.join(arcadia_dir, "")):]
                else:
                    raise Exception("Invalid build path: {}".format(full_path))
                if command == "make":
                    logger.info("Make: %s", build_path)
                    arc.do_build(
                        build_system=arcc.DISTBUILD_BUILD_SYSTEM,
                        source_root=arcadia_dir,
                        targets=[build_path],
                        results_dir=arcadia_dir,
                        clear_build=False,
                    )
                elif command == "package":
                    logger.info("Package: %s", build_path)
                    destination_path = os.path.join(self.package_path, hashlib.md5(full_path + "\n").hexdigest())
                    logger.info("Package destination path: %s", destination_path)
                    old_system_path = os.getcwd()
                    os.makedirs(destination_path)
                    os.chdir(destination_path)
                    try:
                        arc.do_package(
                            build_system=arcc.DISTBUILD_BUILD_SYSTEM,
                            source_root=fake_svn_root,
                            packages=[build_path],
                            clear_build=False,
                        )
                    finally:
                        os.chdir(old_system_path)
                    logger.info("Package destination content: %s", os.listdir(destination_path))
        finally:
            shutil.rmtree(fake_svn_root)

    @property
    def package_path(self):
        return str(self.path("package"))

    def on_execute(self):
        os.environ["NIRVANA_TOKEN"] = self.Parameters.nirvana_token.data()
        os.environ["YT_TOKEN"] = self.Parameters.yt_token.data()
        os.environ["SANDBOX_TOKEN"] = self.Parameters.sandbox_token.data()

        nirvana_make_binary = None
        if self.Parameters.nirvana_make_arcadia_url:
            if self.Parameters.nirvana_make_arcadia_url != self.Parameters.checkout_arcadia_from_url:
                with arc.mount_arc_path(self.Parameters.nirvana_make_arcadia_url) as arcadia_dir:
                    nirvana_make_binary = self.build_nirvana_make(arcadia_dir, copy_file=True)

        with arc.mount_arc_path(self.Parameters.checkout_arcadia_from_url) as arcadia_dir:
            if not nirvana_make_binary:
                nirvana_make_binary = self.build_nirvana_make(arcadia_dir)

            command = [
                nirvana_make_binary,
                "operation_make",
                "--build",
                self.Parameters.operation_path,
                "--root",
                self.Parameters.root_path,
                "--import",
                self.Parameters.import_path,
                "--arcadia_path",
                str(arcadia_dir),
                "--make_command",
                "echo make",
                "--package_command",
                "echo package",
            ]

            logger.info("Running command: %s", command)

            with process_wrapper(self, logger='nirvana_make') as pl:
                build_output = sp.check_output(command, stderr=pl.stderr)

            self.build_operations(arcadia_dir, build_output)

            command = [
                nirvana_make_binary,
                "operation_make",
                "--{}".format(self.Parameters.mode),
                self.Parameters.operation_path,
                "--root",
                self.Parameters.root_path,
                "--import",
                self.Parameters.import_path,
                "--arcadia_path",
                str(arcadia_dir),
                "--sandbox_owner",
                self.owner,
                "--make_command",
                "bash -c 'test -d \"$0\"'",
                "--package_command",
                "bash -c 'cp \"$0/$(md5sum <<< \"$1\" | cut -d \" \" -f 1)\"/* ./' {}".format(pipes.quote(self.package_path)),
            ]

            logger.info("Running command: %s", command)

            with process_wrapper(self, logger='nirvana_make') as pl:
                sp.check_call(command, stdout=pl.stdout, stderr=pl.stderr)

            logger.info("Finished")
