import os
import urllib

import sandbox.common.types.client as ctc

from sandbox import sdk2

from sandbox.common.types.misc import NotExists

from sandbox.projects.common import dolbilka2
from sandbox.projects.dj.server import BasesearchRecommender
from sandbox.projects.tank.load_resources import resources as tank_resources
from sandbox.projects.tank import executor2 as tank_executor


class DjTestBasesearchPerformance(sdk2.Task):
    new_stats_key = "new_stats"

    new_stats_types = (
        ("shooting.rps_0.5", "RPS P50", "{:0.02f}"),
        ("shooting.rps_stddev", "RPS stddev", "{:0.02f}"),
        ("shooting.latency_0.5", "Latency P50", "{:0.02f}"),
        ("shooting.latency_0.95", "Latency P95", "{:0.02f}"),
        ("shooting.latency_0.99", "Latency P99", "{:0.02f}"),
        ("shooting.errors", "Errors", "{}"),
        ("monitoring.cpu_user", "CPU P50", "{:0.02f}"),
    )

    class Parameters(sdk2.Task.Parameters):
        basesearch_resource = sdk2.parameters.Resource(
            'Basesearch basesearch resource',
            required=True
        )
        basesearch_config_resource = sdk2.parameters.Resource(
            'Basesearch config resource',
            required=True
        )
        resource_bundle_resource = sdk2.parameters.Resource(
            'Resource bundle resource',
            required=True
        )
        shard_resource = sdk2.parameters.Resource(
            'Shard resource',
            required=True
        )
        request_count = sdk2.parameters.Integer(
            "Count of requests",
            default=1000,
            required=True
        )
        dolbilka_plan_resource = sdk2.parameters.Resource(
            "Resource with dolbilka plan",
            required=True
        )
        ram = sdk2.parameters.Integer(
            'Ram gb',
            default=10,
            required=True)
        disk_space = sdk2.parameters.Integer(
            'Disk space gb',
            default=20,
            required=True)
        cores = sdk2.parameters.Integer(
            'Cores',
            default=16,
            required=True)
        dolbilka_param = dolbilka2.DolbilkaExecutor2.Parameters
        lunapark_param = tank_executor.LunaparkPlugin.Parameters
        offline_param = tank_executor.OfflinePlugin.Parameters

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC & ctc.Tag.LINUX_PRECISE & ctc.Tag.INTEL_E5_2650

    def on_save(self):
        self.Requirements.ram = self.Parameters.ram * 1024
        self.Requirements.disk_space = self.Parameters.disk_space * 1024
        self.Requirements.cores = self.Parameters.cores

    def on_execute(self):
        self._init_virtualenv()

        params = self.Parameters
        os.mkdir('resources')
        os.symlink(str(sdk2.ResourceData(params.resource_bundle_resource).path),
            os.path.join(os.getcwd(), 'resources/resources'))

        self.basesearch = BasesearchRecommender(
            basesearch_path=str(sdk2.ResourceData(params.basesearch_resource).path),
            basesearch_conf_path=str(sdk2.ResourceData(params.basesearch_config_resource).path),
            shard_path=str(sdk2.ResourceData(params.shard_resource).path)
        )

        self._dolbilo_shoot(self.basesearch.get_port(), params.dolbilka_plan_resource, "perf_test")

    @sdk2.footer()
    def footer(self):
        if getattr(self.Context, self.new_stats_key) is NotExists:
            return "Calculating..."

        stats = getattr(self.Context, self.new_stats_key)

        body = ""
        for key, title, fmt in self.new_stats_types:
            body += "<tr><td style=\"border: 1px; border-style: solid\">{}</td><td style=\"border: 1px; border-style: solid\">{}</td></tr>".format(key, self._format_stats(fmt, stats, key))

        return "<h3>Performance stats</h3><table style=\"border: 1px; border-collapse: collapse; border-style: solid\">{}</table>".format(body)

    def _format_stats(self, fmt, stats, key):
        return fmt.format(stats[key])

    def _shoot(self, executor, variant, autostop_expected=False):
        description = "{}, basesearch {}".format(self.Parameters.description, variant)

        stats_dir = "fire.{}".format(urllib.quote(variant, safe=""))
        stats_resource = sdk2.Resource[tank_resources.YANDEX_TANK_LOGS](self, description, stats_dir)
        stats_resource_data = sdk2.ResourceData(stats_resource)
        stats_resource_data.path.mkdir(0o755, parents=True, exist_ok=True)
        work_dir = str(stats_resource_data.path)

        # Note: cgroup is too restrictive to obtain maximum possible rps
        artifacts_dir = executor.fire(
            work_dir,
            job_name=description,
            cgroup=None,
            autostop_expected=autostop_expected,
        )
        return stats_resource, artifacts_dir

    def _dolbilo_shoot(self, port, plan_resource, variant, augment_url=None):
        stats_resource, artifacts_dir = self._shoot(
            tank_executor.TankExecutor2(
                tank_executor.DolbiloPlugin(self, plan_resource, "localhost", port, augment_url=augment_url),
                *self._default_plugins()
            ),
            variant
        )

        # new stats
        new_stats_results = tank_executor.DolbiloPlugin.get_stats(artifacts_dir)
        new_stats_results.update({
            "report_resource": stats_resource.id,
            "report_path": os.path.relpath(tank_executor.TankExecutor2.get_report_path(artifacts_dir), str(stats_resource.path)),
        })

        lunapark_url = tank_executor.LunaparkPlugin.get_job_url(self)
        if lunapark_url:
            new_stats_results["report_url"] = lunapark_url

        if getattr(self.Context, self.new_stats_key) is NotExists:
            setattr(self.Context, self.new_stats_key, new_stats_results)

    def _default_plugins(self):
        return [
            tank_executor.JsonReportPlugin(self),
            tank_executor.LunaparkPlugin(self),
            tank_executor.TelegrafPlugin(self),
            tank_executor.OfflinePlugin(self)
        ]

    def _init_virtualenv(self, tank_resource_type=tank_resources.YANDEX_TANK_VIRTUALENV_19):
        tank_executor.TankExecutor2.init_virtualenv(self, tank_resource_type=tank_resource_type);
