import logging
import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

from sandbox import sdk2
from sandbox.common.types.misc import NotExists
from sandbox.projects.common import binary_task


def create_session():
    session = requests.Session()
    retry = Retry(
        total=3,
        backoff_factor=0.3,
    )
    adapter = HTTPAdapter(max_retries=retry)
    session.mount('http://', adapter)
    return session


class DjTestFreshProfileServer(binary_task.LastBinaryTaskRelease, sdk2.Task):
    stats_key = 'stats'

    stats_types = (
        ("latency_avg", "Average latency", "{:0.02f}"),
        ("latency_p50", "Median latency", "{:0.02f}"),
        ("latency_p99", "P99 latency", "{:0.02f}"),
        ("failed_requests_ratio", "Ratio of failed requests", "{:0.02f}"),
        ("max_rss", "Max memory, MB", "{}"),
    )

    class Parameters(sdk2.Task.Parameters):
        fps_resource = sdk2.parameters.Resource('Fresh profile server binary', required=True)

        config_resource = sdk2.parameters.Resource('Fresh profile server config', required=True)

        rules_resource = sdk2.parameters.Resource('Rules config', required=True)

        server_config_resource = sdk2.parameters.Resource('Server config', required=True)

        custom_files = sdk2.parameters.Dict('Additional resources (id - path)')

        shard_dir = sdk2.parameters.String('Shard dir', required=False)

        queries_resource = sdk2.parameters.Resource('Queries file', required=True)

        ext_params = binary_task.binary_release_parameters(stable=True)

    def on_execute(self):
        from dj.lib.processing.online.fresh_profile_server.proto import config_pb2
        import google.protobuf.text_format
        from fresh_profile_server import FreshProfileServer

        import numpy as np
        import psutil
        from urlparse import urlparse

        params = self.Parameters
        binary_path = str(sdk2.ResourceData(params.fps_resource).path)
        config_path = str(sdk2.ResourceData(params.config_resource).path)
        rules_path = str(sdk2.ResourceData(params.rules_resource).path)
        server_config_path = str(sdk2.ResourceData(params.server_config_resource).path)
        self.get_custom_files()

        config_proto = config_pb2.TFreshProfileServerConfig()

        with open(config_path, 'r') as f:
            google.protobuf.text_format.Parse(f.read(), config_proto)
            api_path = config_proto.ServerConfig.Path

        fps = FreshProfileServer(binary_path, config_path, rules_path, server_config_path, params.shard_dir)
        process = psutil.Process(fps.process.pid)
        session = create_session()

        latency = []
        n_failed_queries, n_queries = 0, 0
        max_rss = 0
        for url in open(str(sdk2.ResourceData(params.queries_resource).path), 'r'):
            url = url.strip()
            if not url:
                continue
            response = session.get('http://localhost:{}/{}?{}'.format(fps.get_port(), api_path, urlparse(url).query))
            if response.ok:
                latency.append(response.elapsed.total_seconds() * 1000)
                max_rss = max(max_rss, process.memory_info().rss)
            else:
                n_failed_queries += 1
            n_queries += 1
        logging.info('Finished shooting')

        stats = {
            'latency_avg': np.average(latency),
            'latency_p50': np.median(latency),
            'latency_p99': np.percentile(latency, 99),
            'failed_requests_ratio': 1. * n_failed_queries / n_queries,
            'max_rss': max_rss / (1024 * 1024),
        }

        if getattr(self.Context, self.stats_key) is NotExists:
            setattr(self.Context, self.stats_key, stats)

    def get_custom_files(self):
        import os
        from shutil import copy
        import tarfile

        for id, target in self.Parameters.custom_files.items():
            id = int(id)  # expect resource ids
            path = str(sdk2.ResourceData(sdk2.Resource.find(id=id).first()).path)

            if path.endswith('.tar.gz'):
                logging.info('Unpacking package {} to {}'.format(path, target))
                with tarfile.open(path, 'r:gz') as tar:
                    tar.extractall(path=target)
                continue

            dirname = os.path.dirname(target)
            if dirname:
                try:
                    os.makedirs(dirname)
                    logging.info('Created dir {}'.format(dirname))
                except OSError as err:
                    # skip if already exists (exist_ok=True in py3)
                    print(err)
            if target:
                copy(path, target)
            logging.info('Copied {} to {}'.format(path, target))

    @sdk2.footer()
    def footer(self):
        if getattr(self.Context, self.stats_key) is NotExists:
            return "Calculating..."

        stats = getattr(self.Context, self.stats_key)

        body = ""
        for key, title, fmt in self.stats_types:
            body += "<tr><td style=\"border: 1px; border-style: solid\">{}</td><td style=\"border: 1px; border-style: solid\">{}</td></tr>".format(title, fmt.format(stats[key]))

        return "<h3>Stats</h3><table style=\"border: 1px; border-collapse: collapse; border-style: solid\">{}</table>".format(body)
