import os
import urllib
import logging

import sandbox.common.types.client as ctc

from sandbox import sdk2

from sandbox.common.types.misc import NotExists

from sandbox.projects.common import dolbilka2
from sandbox.projects.common.arcadia import sdk as arc
from sandbox.projects.common.constants import constants as arcc
from sandbox.projects.dj.server import BasesearchRecommender, MiddlesearchRecommender
from sandbox.projects.tank.load_resources import resources as tank_resources
from sandbox.projects.tank import executor2 as tank_executor

from sandbox.sdk2.helpers import subprocess as sp

import shutil


BASE_SHARD_FOLDER = "saas_shard"

CATENGINE_RESOURCE_FOLDER = "catengine"
DSSM_RESOURCE_FOLDER = "dssm"
EXTRA_RESOURCE_FOLDER = "extra"
FORMULA_RESOURCE_FOLDER = "formulas"
CONFIG_FOLDER = "configs"

logger = logging.getLogger("DjTestSaasMarketPerformance")


class DjTestSaasMarketPerformance(sdk2.Task):
    new_stats_key = "new_stats"

    new_stats_types = (
        ("shooting.rps_0.5", "RPS P50", "{:0.02f}"),
        ("shooting.rps_stddev", "RPS stddev", "{:0.02f}"),
        ("shooting.latency_0.5", "Latency P50", "{:0.02f}"),
        ("shooting.latency_0.95", "Latency P95", "{:0.02f}"),
        ("shooting.latency_0.99", "Latency P99", "{:0.02f}"),
        ("shooting.errors", "Errors", "{}"),
        ("monitoring.cpu_user", "CPU P50", "{:0.02f}"),
    )

    class Parameters(sdk2.Task.Parameters):
        config_folder_path = sdk2.parameters.String(
            "Config folder path",
            required=False,
            default="dj/services/market/configs_generator"
        )
        arcadia_revision = sdk2.parameters.String("Revision", required=True)

        basesearch_path = sdk2.parameters.String(
            "Basesearch tool path",
            required=False,
            default="dj/tools/basesearch/basesearch"
        )
        basesearch_shard_resource = sdk2.parameters.Resource(
            'Shard resource',
            required=True
        )

        metasearch_path = sdk2.parameters.String(
            "Middlesearch tool path",
            required=False,
            default="dj/tools/middlesearch/middlesearch"
        )

        catengine_resource = sdk2.parameters.Resource(
            'Catengine resource',
            required=True
        )
        dssm_resource = sdk2.parameters.Resource(
            'Dssm resource',
            required=True
        )
        extra_resource = sdk2.parameters.Resource(
            'Extra resource',
            required=True
        )
        formula_resource = sdk2.parameters.Resource(
            'Formulas resource',
            required=True
        )
        request_count = sdk2.parameters.Integer(
            "Count of requests",
            default=1000,
            required=True
        )
        dolbilka_plan_resource = sdk2.parameters.Resource(
            "Resource with dolbilka plan",
            required=True
        )
        ram = sdk2.parameters.Integer(
            'Ram gb',
            default=10,
            required=True)
        disk_space = sdk2.parameters.Integer(
            'Disk space gb',
            default=20,
            required=True)
        cores = sdk2.parameters.Integer(
            'Cores',
            default=16,
            required=True)
        dolbilka_param = dolbilka2.DolbilkaExecutor2.Parameters
        lunapark_param = tank_executor.LunaparkPlugin.Parameters
        offline_param = tank_executor.OfflinePlugin.Parameters

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC & ctc.Tag.LINUX_PRECISE & ctc.Tag.INTEL_E5_2650

    def on_save(self):
        self.Requirements.ram = self.Parameters.ram * 1024
        self.Requirements.disk_space = self.Parameters.disk_space * 1024
        self.Requirements.cores = self.Parameters.cores

    def prepare_binaries(self, arcadia_dir):
        logger.info("Start to build basesearch")
        arc.do_build(
            build_system=arcc.SEMI_DISTBUILD_BUILD_SYSTEM,
            source_root=arcadia_dir,
            targets=[os.path.dirname(self.Parameters.basesearch_path)],
            results_dir=arcadia_dir,
            clear_build=False,
        )
        logger.info("End to make basesearch")

        logger.info("Start to build middlesearch")
        arc.do_build(
            build_system=arcc.SEMI_DISTBUILD_BUILD_SYSTEM,
            source_root=arcadia_dir,
            targets=[os.path.dirname(self.Parameters.metasearch_path)],
            results_dir=arcadia_dir,
            clear_build=False,
        )
        logger.info("End to make middlesearch")
        return os.path.join(arcadia_dir, self.Parameters.basesearch_path), os.path.join(arcadia_dir, self.Parameters.metasearch_path)

    def build_saas_configs(self, arcadia_dir):
        logger.info("Start to make configs")
        configs_generator_dir = os.path.join(arcadia_dir, 'dj/services/market/configs_generator')
        configs_dir = os.path.join(configs_generator_dir, 'generated')
        arc.do_build(
            build_system=arcc.SEMI_DISTBUILD_BUILD_SYSTEM,
            source_root=arcadia_dir,
            targets=[configs_generator_dir],
            results_dir=arcadia_dir,
            clear_build=False,
        )
        cmd = [
            os.path.join(configs_generator_dir, 'configs_generator'),
            '--result-dir', configs_dir,
        ]
        process = sp.Popen(cmd)
        process.wait()
        shutil.copytree(configs_dir, os.path.join(os.getcwd(), CONFIG_FOLDER))

    def populate_saas_resources(self):
        os.symlink(str(sdk2.ResourceData(self.Parameters.basesearch_shard_resource).path),
                   os.path.join(os.getcwd(), BASE_SHARD_FOLDER))

        os.symlink(str(sdk2.ResourceData(self.Parameters.catengine_resource).path),
                   os.path.join(os.getcwd(), CATENGINE_RESOURCE_FOLDER))
        os.symlink(str(sdk2.ResourceData(self.Parameters.dssm_resource).path),
                   os.path.join(os.getcwd(), DSSM_RESOURCE_FOLDER))
        os.symlink(str(sdk2.ResourceData(self.Parameters.extra_resource).path),
                   os.path.join(os.getcwd(), EXTRA_RESOURCE_FOLDER))
        os.symlink(str(sdk2.ResourceData(self.Parameters.formula_resource).path),
                   os.path.join(os.getcwd(), FORMULA_RESOURCE_FOLDER))

    def write_mmeta_config(self, basesearch_port):
        meta_config_template = '''
            <Server>
                Port 8034
                Threads 1
                QueueSize 100
                AdminThreads 1
                EventLog ./middlesearch8034_event.log
                Compression true
                Connections 5000
                ConfigVersion stable-96-r30
                AppHostOptions Port=+1, Threads=8
                ServerLog ./middlesearch8034_server.log
                LogExceptions True
                LoadLog ./middlesearch8034_load.log
            </Server>
            <Collection autostart="must" meta="yes" id="yandsearch">
                ServiceName DJ_MMETA
                RearrangeDataDir ./
                RearrangeIndexDir ./
                ReArrangeOptions DJ()
                MetaSearchOptions AsyncSearch=1, OneStepQuery=1
                MetaRankingOptions Enabled=no, Default=no, Groups=60, Docs=60, GoodDocs=yes, UseGroups=yes
                MergeOptions SkipSameDocids=no
                CommonSourceOptions EnableIpV6=yes, EnableUnresolvedHosts=yes, MaxAttempts=4, ProtocolType=proto, OneStepQuery=1
                SearchCgi allfctrs=da
                ServiceDiscoveryOptions Enabled=true, ClientName=dj
                <SearchSource>
                    CgiSearchPrefix http://localhost:%s/yandsearch
                    ServerDescr static
                </SearchSource>
                <UserParams>
                    RecommenderSystemConfig %s
                    CustomResourceDirectory %s
                    CustomModelsDirectory %s
                    StrongRecommenderInit
                </UserParams>
            </Collection>
        ''' % (basesearch_port, os.path.join(os.getcwd(), CONFIG_FOLDER, "recommender_saas_meta_mock.pbtxt"),
               os.path.join(os.getcwd(), CONFIG_FOLDER), os.path.join(os.getcwd(), DSSM_RESOURCE_FOLDER))

        mmeta_config_path = os.path.join(os.getcwd(), "metasearch.cfg")
        with open(mmeta_config_path, "w") as f:
            f.write(meta_config_template)
        return mmeta_config_path

    def write_base_config(self):
        base_config_template = '''
            <Collection id="yandsearch" autostart="must" class="DJ">
                IndexDir %s
                RequestThreads 5
                RequestQueueSize 5
                <UserParams>
                    RecommenderSystemConfig %s
                    CustomResourceDirectory %s
                    CustomResourceMultiPath %s;%s;%s
                </UserParams>
            </Collection>
            <Server>
                Port 8000
                Threads 2
                QueueSize 5
                Connections 300
                Compression true
             </Server>
        ''' % (os.path.join(os.getcwd(), BASE_SHARD_FOLDER),
                os.path.join(os.getcwd(), CONFIG_FOLDER, "recommender_saas_base_mock.pbtxt"),
                os.path.join(os.getcwd(), CONFIG_FOLDER),
                os.path.join(os.getcwd(), CATENGINE_RESOURCE_FOLDER),
                os.path.join(os.getcwd(), FORMULA_RESOURCE_FOLDER),
                os.path.join(os.getcwd(), EXTRA_RESOURCE_FOLDER))

        base_config_path = os.path.join(os.getcwd(), "basesearch.cfg")
        with open(base_config_path, "w") as f:
            f.write(base_config_template)
        return base_config_path

    def on_execute(self):
        self._init_virtualenv()

        with arc.mount_arc_path('{}@{}'.format("arcadia:/arc/trunk/arcadia", self.Parameters.arcadia_revision)) as arcadia_dir:
            basesearch, middlesearch = self.prepare_binaries(arcadia_dir)
            logger.info("Start config preparation")

            self.populate_saas_resources()
            self.build_saas_configs(arcadia_dir)

            basesearch_config_path = self.write_base_config()
            self.basesearch = BasesearchRecommender(
                basesearch_path=basesearch,
                basesearch_conf_path=basesearch_config_path,
                shard_path=os.path.join(os.getcwd(), BASE_SHARD_FOLDER)
            )

            middlesearch_config_path = self.write_mmeta_config(self.basesearch.get_port())
            self.middlesearch = MiddlesearchRecommender(
                middlesearch_path=middlesearch,
                middlesearch_conf_path=middlesearch_config_path
            )

        self._dolbilo_shoot(self.middlesearch.get_port(), self.Parameters.dolbilka_plan_resource, "perf_test")

    @sdk2.footer()
    def footer(self):
        if getattr(self.Context, self.new_stats_key) is NotExists:
            return "Calculating..."

        stats = getattr(self.Context, self.new_stats_key)

        body = ""
        for key, title, fmt in self.new_stats_types:
            body += "<tr><td style=\"border: 1px; border-style: solid\">{}</td><td style=\"border: 1px; border-style: solid\">{}</td></tr>".format(key, self._format_stats(fmt, stats, key))

        return "<h3>Performance stats</h3><table style=\"border: 1px; border-collapse: collapse; border-style: solid\">{}</table>".format(body)

    def _format_stats(self, fmt, stats, key):
        return fmt.format(stats[key])

    def _shoot(self, executor, variant, autostop_expected=False):
        description = "{}, basesearch {}".format(self.Parameters.description, variant)

        stats_dir = "fire.{}".format(urllib.quote(variant, safe=""))
        stats_resource = sdk2.Resource[tank_resources.YANDEX_TANK_LOGS](self, description, stats_dir)
        stats_resource_data = sdk2.ResourceData(stats_resource)
        stats_resource_data.path.mkdir(0o755, parents=True, exist_ok=True)
        work_dir = str(stats_resource_data.path)

        # Note: cgroup is too restrictive to obtain maximum possible rps
        artifacts_dir = executor.fire(
            work_dir,
            job_name=description,
            cgroup=None,
            autostop_expected=autostop_expected,
        )
        return stats_resource, artifacts_dir

    def _dolbilo_shoot(self, port, plan_resource, variant, augment_url=None):
        stats_resource, artifacts_dir = self._shoot(
            tank_executor.TankExecutor2(
                tank_executor.DolbiloPlugin(self, plan_resource, "localhost", port, augment_url=augment_url),
                *self._default_plugins()
            ),
            variant
        )

        # new stats
        new_stats_results = tank_executor.DolbiloPlugin.get_stats(artifacts_dir)
        new_stats_results.update({
            "report_resource": stats_resource.id,
            "report_path": os.path.relpath(tank_executor.TankExecutor2.get_report_path(artifacts_dir), str(stats_resource.path)),
        })

        lunapark_url = tank_executor.LunaparkPlugin.get_job_url(self)
        if lunapark_url:
            new_stats_results["report_url"] = lunapark_url

        if getattr(self.Context, self.new_stats_key) is NotExists:
            setattr(self.Context, self.new_stats_key, new_stats_results)

    def _default_plugins(self):
        return [
            tank_executor.JsonReportPlugin(self),
            tank_executor.LunaparkPlugin(self),
            tank_executor.TelegrafPlugin(self),
            tank_executor.OfflinePlugin(self)
        ]

    def _init_virtualenv(self, tank_resource_type=tank_resources.YANDEX_TANK_VIRTUALENV_19):
        tank_executor.TankExecutor2.init_virtualenv(self, tank_resource_type=tank_resource_type)
