from sandbox import sdk2
from sandbox.sandboxsdk import environments

from sandbox.projects.dj.rthub.resources import DjRthubPackage
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper

import sandbox.common.types.task as ctt
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.task_env as rm_task_env
import sandbox.projects.release_machine.components.all as rm_comp

from acceptance import AcceptanceProcess

import json
import logging
import os
import tarfile


class DjRthubAcceptance(sdk2.Task):
    """ Task for dj rthub releases acceptance """

    class Requirements(sdk2.Task.Requirements):
        environments = (
            rm_task_env.TaskRequirements.startrek_client,
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment("yandex-yt-yson-bindings-skynet"),
            environments.PipEnvironment('yandex-yt-transfer-manager-client')
        )
        client_tags = rm_task_env.TaskTags.startrek_client


    class Parameters(sdk2.Task.Parameters):
        package_resource = sdk2.parameters.Resource(
            "Resource with DJ RTHub package to test",
            resource_type=DjRthubPackage,
            required=True
        )
        nirvana_token_vault = sdk2.parameters.String(
            "Nirvana token vault",
            required=True,
            default='robot_dj_unity_nirvana_token'
        )
        component_name = sdk2.parameters.String(
            "Release machine component name for startrek ticket update",
            default='dj_rthub'
        )
        release_number = sdk2.parameters.Integer(
            "Release number for startrek ticket update",
            default=None
        )
        kill_timeout = 24 * 3600


    def on_execute(self):
        with self.memoize_stage.start_process:
            acceptance_process = AcceptanceProcess(
                self.prod_package_resource_id(),
                self.release_package_id(),
                self.Parameters.nirvana_token_vault,
                self.Parameters.release_number
            )
            acceptance_process.start()
            self.Context.acceptance_process = acceptance_process.to_json()

        acceptance_process = AcceptanceProcess.from_json(self.Context.acceptance_process)
        if not acceptance_process.check_finished():
            raise sdk2.WaitTime(10 * 60) # 10 min

        diff_path = "./dj_rthub_diff.tar"
        acceptance_process.download_diff(diff_path)

        st_helper = None
        if self.Parameters.component_name is not None and self.Parameters.component_name:
            st_helper = STHelper(sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME))

        with tarfile.open(diff_path, "r:gz") as tar:
            for item in tar.getmembers():
                if item.isfile():
                    diff = json.load(tar.extractfile(item))
                    self.publish_diff(diff, st_helper)


    def prod_package_resource_id(self):
        res = self.find_production_package_resource()
        if res is not None:
            return str(res.id)
        else:
            return "1708351300" # initial release


    def release_package_id(self):
        return str(self.Parameters.package_resource.id)


    def publish_diff(self, diff, st_helper):
        test_id = diff["id"]
        diffs = diff["diffs"]
        message = "Test " + test_id
        if not diffs:
            message += " has no diffs"
        else:
            message += " has diffs in topics:"
            for topic_diff in diffs:
                message += "\n" + topic_diff["topic"]
                message += "\n" + "<{Show diff\n" + topic_diff["diff"] + "}>"

        if st_helper is None:
            logging.info(message)
        else:
            c_info = rm_comp.COMPONENTS[self.Parameters.component_name]()
            st_helper.comment(self.Parameters.release_number, message, c_info)


    @staticmethod
    def find_production_package_resource():
        logging.info('Searching for production package')
        res = sdk2.Resource.find(
            type=DjRthubPackage,
            attrs={
                "released": ctt.ReleaseStatus.STABLE
            }
        ).order(-sdk2.Resource.id).first()
        if not res:
            logging.info("Cannot find released package")
            return None

        logging.info('Production package: {}'.format(res.id))
        return res
