from sandbox import common
from sandbox import sdk2

from sandbox.projects.dj.unity.utils import nirvana

import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.components.all as rm_comp

import json
import logging
import os
import requests
import shutil

DRAFT = "draft"
EXECUTING = "executing"
FINISHED = "finished"
BROKEN = "broken"

class AcceptanceProcess(object):
    def __init__(
        self,
        prod_package_resource_id,
        release_package_resource_id,
        nirvana_token_vault,
        release_number,
        workflow_instance=None
    ):
        self.prod_package_resource_id = prod_package_resource_id
        self.release_package_resource_id = release_package_resource_id
        self.nirvana_token_vault = nirvana_token_vault
        self.nirvana_client = self.create_nirvana_client()
        self.release_number = release_number
        self.workflow_instance = workflow_instance


    def to_json(self):
        return {
            "prod_package_resource_id": self.prod_package_resource_id,
            "release_package_resource_id": self.release_package_resource_id,
            "nirvana_token_vault": self.nirvana_token_vault,
            "release_number": self.release_number,
            "workflow_instance":
                self.workflow_instance.to_json() if self.workflow_instance is not None else None
        }

    @classmethod
    def from_json(cls, json):
        workflow_instance = None
        workflow_instance_json = json["workflow_instance"]
        if workflow_instance_json is not None:
            workflow_instance = cls.WorkflowInstance.from_json(json["workflow_instance"])

        return cls(
            json["prod_package_resource_id"],
            json["release_package_resource_id"],
            json["nirvana_token_vault"],
            json["release_number"],
            workflow_instance
        )


    def start(self):
        self.workflow_instance = self.start_new_instance()


    def check_finished(self):
        if self.workflow_instance.state != EXECUTING :
            raise common.errors.TaskFailure("Bad process state: {}".format(self.workflow_instance.state))

        instance_state = self.nirvana_client.get_workflow_instance_state(
            self.workflow_instance.workflow_id,
            self.workflow_instance.instance_id
        )
        if instance_state['status'] == 'completed':
            if instance_state['result'] != 'success':
                self.workflow_instance.state = BROKEN
                raise common.errors.TaskFailure(
                    'Instance {}/{} failed. Status: {}'.format(
                        self.workflow_instance.workflow_id,
                        self.workflow_instance.instance_id,
                        instance_state['result']
                    )
                )
            self.workflow_instance.state = FINISHED
            logging.info(
                'Instance {}/{} finished successfully.'.format(
                    self.workflow_instance.workflow_id,
                    self.workflow_instance.instance_id
                )
            )
            return True
        return False


    def download_diff(self, local_path):
        if self.workflow_instance.state != FINISHED:
            raise common.errors.TaskFailure("Bad process state: {}".format(self.workflow_instance.state))

        result = {}
        block_results = self.nirvana_client.get_blocks_result(
            self.workflow_instance.workflow_id,
            self.workflow_instance.instance_id,
            self.create_blocks_description([self.workflow_instance.result_block_id])
        )
        for output in block_results[0]["results"]:
            if output["endpoint"] == self.workflow_instance.result_output_id:
                storage_url = output["directStoragePath"]
                with requests.get(storage_url, verify=False, stream=True) as r:
                    r.raise_for_status()
                    with open(local_path, "wb") as f:
                        shutil.copyfileobj(r.raw, f)


    class WorkflowInstance(object):
        def __init__(
            self,
            workflow_id,
            template_instance_id,
            result_block_id,
            result_output_id,
            global_params,
            instance_id,
            state
        ):
            self.workflow_id = workflow_id
            self.template_instance_id = template_instance_id
            self.result_block_id = result_block_id
            self.result_output_id = result_output_id
            self.global_params = global_params
            self.instance_id = instance_id
            self.state = state

        @classmethod
        def from_json(cls, json):
            return cls(
                json["workflow_id"],
                json["template_instance_id"],
                json["result_block_id"],
                json["result_output_id"],
                json["global_params"],
                json["instance_id"],
                json["state"]
            )

        def to_json(self):
            return {
                "workflow_id": self.workflow_id,
                "template_instance_id": self.template_instance_id,
                "result_block_id": self.result_block_id,
                "result_output_id": self.result_output_id,
                "global_params": self.global_params,
                "instance_id": self.instance_id,
                "state": self.state
            }


    def start_new_instance(self):
        config = json.loads(
            """
            {
                "workflow_id": "1f2980cf-ffbe-46fc-a7c0-8fcb5cfeff09",
                "template_instance_id": "7f2fc701-fb57-40df-b010-a67c3d679e55",
                "result_block_id": "package_diff",
                "result_output_id": "diffs",
                "global_param_keys": {
                    "prod_package_resource_id": "prod-package-resource-id",
                    "release_package_resource_id": "release-package-resource-id"
                }
            }
            """
        )

        logging.info('Creating new workflow instance')
        instance_id = self.nirvana_client.clone_workflow_instance(
            config["workflow_id"],
            config["template_instance_id"]
        )

        instance = AcceptanceProcess.WorkflowInstance(
            config["workflow_id"],
            config["template_instance_id"],
            config["result_block_id"],
            config["result_output_id"],
            self.build_workflow_params(
                config["global_param_keys"],
                prod_package_resource_id=self.prod_package_resource_id,
                release_package_resource_id=self.release_package_resource_id
            ),
            instance_id,
            DRAFT
        )

        self.nirvana_client.add_comment_to_workflow_instance(
            instance.instance_id,
            'DJ Rthub acceptance graph for release {}'.format(self.release_number)
        )
        logging.info('New instance: {}/{}'.format(instance.workflow_id, instance.instance_id))

        logging.info('Setting new params: {}'.format(instance.global_params))
        self.nirvana_client.set_workflow_parameters(
            instance.workflow_id,
            instance.instance_id,
            instance.global_params
        )
        self.nirvana_client.approve_workflow_instance(
            instance.workflow_id,
            instance.instance_id
        )

        logging.info('Starting instance')
        self.nirvana_client.start_workflow_instance(
            instance.workflow_id,
            instance.instance_id
        )
        instance.state = EXECUTING
        return instance


    def build_workflow_params(self, key_map, **params):
        workflow_params = []
        for name, value in params.items():
            workflow_params.append(
                {
                    "parameter": key_map[name],
                    "value": value
                }
            )
        return workflow_params


    def create_nirvana_client(self):
        logging.info('Creating nirvana client')
        nirvana_token = sdk2.Vault.data(self.nirvana_token_vault)
        return nirvana.NirvanaClient(nirvana_token)


    def create_blocks_description(self, block_ids):
        blocks = []
        for b in block_ids:
            blocks.append({
                "code": b
            })
        return blocks
