import logging
import os
import tarfile

from sandbox.projects.dj.services.chz.specs import const
from sandbox.projects.dj.services.chz.specs import utils


def untar_to(archive_path, output_dir_path):
    logging.debug('Untaring %s to %s', archive_path, output_dir_path)
    with tarfile.open(archive_path, "r:gz") as tar:
        tar.extractall(output_dir_path)


def untar_resource(_spec_field, resource_path, output_dir_path):
    untar_to(str(resource_path), output_dir_path)


def link_into_dir(src, output_dir_path):
    if not os.path.exists(output_dir_path):
        os.makedirs(output_dir_path)

    dst = os.path.join(output_dir_path, os.path.basename(src))
    logging.debug('Linking %s to %s', src, dst)
    os.symlink(src, dst)


def link_resource(spec_field, resource_path, output_dir_path):
    output_dir = output_dir_path
    if resource_path.is_dir():
        output_dir = os.path.join(
            output_dir_path,
            spec_field.name
        )

    link_into_dir(str(resource_path), output_dir)


def make_untar_step(spec_field, resource_path):
    def result(wd):
        untar_resource(spec_field, resource_path, wd)

    return result


def make_link_step(spec_field, resource_path):
    def result(wd):
        link_resource(spec_field, resource_path, wd)

    return result


def spec_to_deploy_steps(spec_object, skip_nones=False):
    steps = []

    objects_to_untar = utils.select_from_spec_by_metadata_flag(
        spec_object,
        metadata_flag=const.UNTAR_ATTR
    )

    for spec_field, resource_path in objects_to_untar:
        if not skip_nones or resource_path is not None:
            logging.debug('Selected for untaring: %s, %s', spec_field.name, resource_path)
            steps.append(make_untar_step(spec_field, resource_path))

    objects_to_link = utils.select_from_spec_by_metadata_flag(
        spec_object,
        metadata_flag=const.LINK_ATTR
    )

    for spec_field, resource_path in objects_to_link:
        if not skip_nones or resource_path is not None:
            logging.debug('Selected for linking: %s, %s', spec_field.name, resource_path)
            steps.append(make_link_step(spec_field, resource_path))

    return steps
