import subprocess as sp


class ChzRecommenderBase:
    def __init__(
        self,
        binary_path,
        http_port,
        config_path,
        working_dir,
        process_log=None,
    ):
        self.binary_path = binary_path
        self.http_port = http_port
        self.config_path = config_path
        self.working_dir = working_dir
        self.process_log = process_log
        self.process = None
        self.args = None

    def run(self):
        if self.process:
            return False

        self._make_args()
        self._make_popen_args()
        self._launch_process()
        return True

    def terminate(self):
        if self.process:
            self.process.terminate()

    def __enter__(self):
        self.run()
        return self

    def __exit__(self, exc_type, exc_value, exc_traceback):
        self.terminate()

    def _make_args(self):
        self.args = [self.binary_path]
        self.args.extend(self._make_required_args())

    def _make_popen_args(self):
        self.popen_args = {}
        self.popen_args.update(self._make_popen_required_args())
        self.popen_args.update(self._make_popen_optional_args())

    def _make_required_args(self):
        return [
            "--config", self.config_path,
            "--port", str(self.http_port),
        ]

    def _make_popen_required_args(self):
        return {
            "stderr": sp.STDOUT,
            "cwd": self.working_dir,
        }

    def _make_popen_optional_args(self):
        args = {}
        if self.process_log:
            args["stdout"] = self.process_log.stdout
        return args

    def _launch_process(self):
        raise NotImplementedError()
