import itertools
import logging
import shutil
import tempfile

from sandbox.sdk2 import Task
from sandbox.sdk2 import ResourceData
from sandbox.sdk2 import Vault

from sandbox.sdk2.helpers import ProcessLog
from sandbox.sdk2.helpers import subprocess

from sandbox.sdk2.parameters import Integer
from sandbox.sdk2.parameters import Dict
from sandbox.sdk2.parameters import String
from sandbox.sdk2.parameters import List
from sandbox.sdk2.parameters import Bool

from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.common import binary_task
from sandbox.projects.common import dolbilka

from sandbox.projects.resource_types import APP_HOST_TOOLS_BUNDLE

from sandbox.projects.dj.services.entity import helpers

from sandbox.projects.dj.services.entity.common import ATTR_REQUESTS_COUNT
from sandbox.projects.dj.services.entity.common import ATTR_EXPERIMENT
from sandbox.projects.dj.services.entity.common import ATTR_AMMO_TYPE
from sandbox.projects.dj.services.entity.common import AMMO_TYPE_APPHOST
from sandbox.projects.dj.services.entity.common import AMMO_TYPE_HTTP

from sandbox.projects.dj.services.entity.resources import EntityRecommenderShootingPlan
from sandbox.projects.dj.services.entity.resources import EntityRecommenderTextQueries


FILE_SUFFIX_PLAIN_TEXT_QUERIES = "queries.txt"
FILE_SUFFIX_BASESEARCH_PLAN = "plan.phantom"


class GenerateEntityRecommenderRequests(binary_task.LastBinaryTaskRelease, Task):
    class Parameters(Task.Parameters):
        yt_proxy = String("YT server name", default="hahn")
        yt_token_vault = String("YT token vault", required=True)
        log_path = String("YT table dir or with entity recommender log tables", required=True)
        requests_count = Integer("Requests count per file", default=100000)
        additional_attributes = Dict("Additional attributes", default={})
        rps = Integer("RPS", default=10)
        uri_prefixes = List("Uri prefix for log filtering", default=[])
        experiments = List("Experiments", default=[])
        shuffle = Bool("Shuffle requests", default=False)
        generate_apphost_ammo = Bool("Generate apphost ammo (leave unchecked for http)", default=False)

        ext_params = binary_task.binary_release_parameters(stable=True)

    class Requirements(Task.Requirements):
       environments = (
           PipEnvironment('yandex-yt'),
           PipEnvironment("yandex-yt-yson-bindings-skynet")
       )

    def make_plain_text_queries(self):
        parameters = self.Parameters
        logging.info('Started worker...')

        from . import worker

        ptq_maker = worker.PlainTextQueriesMaker(
            cluster=parameters.yt_proxy,
            token=Vault.data(parameters.yt_token_vault),
            file_suffix=FILE_SUFFIX_PLAIN_TEXT_QUERIES,
            requests_count=parameters.requests_count,
            log_path=parameters.log_path,
            uri_prefixes=parameters.uri_prefixes,
            experiments=helpers.generate_ammo_update_experiments_list(parameters.experiments),
            shuffle=parameters.shuffle,
            generate_apphost_ammo=parameters.generate_apphost_ammo
        )
        ptq_maker.run()

        logging.info('Finished worker!')

    def convert_apphost_queries_to_binary_format(self, apphost_queries_file_path):
        logging.info('Started converting queries to binary format...')

        app_host_tools_bundle = helpers.init_resource(
            APP_HOST_TOOLS_BUNDLE,
            resource_attributes = {"released": "stable"}
        )
        app_host_tools_bundle_data = ResourceData(app_host_tools_bundle)

        # move: apphost_queries_file_path -> temp_file
        with tempfile.NamedTemporaryFile() as temp_file:
            shutil.move(apphost_queries_file_path, temp_file.name)

            # generate tank ammo: temp_file -> apphost_queries_file_path
            cmd = helpers.CmdArgsBuilder() \
                    .add_positional_argument(helpers.resource_data_path(app_host_tools_bundle_data) + "/make_tank_ammo") \
                    .add_argument('-i', temp_file.name) \
                    .add_argument('-o', apphost_queries_file_path) \
                    .build()

            logging.info("Starting Make Tank Ammo")
            with ProcessLog(self, logger=logging.getLogger("make_tank_ammo")) as pl:
                return_code = subprocess.Popen(cmd, stdout=pl.stdout, stderr=subprocess.STDOUT).wait()
                if return_code == 0:
                    logging.info("Make Tank Ammo finished successfully")
                else:
                    logging.error('Subprocess ended with error: %s', return_code)
                    raise Exception("Subprocess failed!")

        logging.info('Finished converting queries to binary format!')

    def make_plans(self):
        logging.info('Started making plans...')

        parameters = self.Parameters
        for exp in helpers.generate_ammo_update_experiments_list(parameters.experiments):
            resource_attrs = {
                ATTR_EXPERIMENT: exp,
                ATTR_REQUESTS_COUNT: parameters.requests_count,
                ATTR_AMMO_TYPE: AMMO_TYPE_APPHOST if parameters.generate_apphost_ammo else AMMO_TYPE_HTTP
            }
            resource_attrs.update(parameters.additional_attributes)

            queries_file_path = helpers.make_file_name_for_experiment(exp, FILE_SUFFIX_PLAIN_TEXT_QUERIES)
            if parameters.generate_apphost_ammo:
                self.convert_apphost_queries_to_binary_format(queries_file_path)
            queries_resource = EntityRecommenderTextQueries(
                self, 'Requests with `{}` experiment in plain text format'.format(exp),
                queries_file_path,
                **resource_attrs
            )
            queries_resource_data = ResourceData(queries_resource)
            queries_resource_data_path = helpers.resource_data_path(queries_resource_data)
            shutil.move(queries_file_path, queries_resource_data_path)
            queries_resource_data.ready()

            basesearch_plan_resource = EntityRecommenderShootingPlan(
                self, 'Requests with `{}` experiment in dolbilka plan format'.format(exp),
                helpers.make_file_name_for_experiment(exp, FILE_SUFFIX_BASESEARCH_PLAN),
                **resource_attrs
            )
            plan_resource_data = ResourceData(basesearch_plan_resource)
            plan_resource_data_path = helpers.resource_data_path(plan_resource_data)
            dolbilka.convert_queries_to_plan(
                queries_resource_data_path, plan_resource_data_path,
                rps=self.Parameters.rps, loader_type='phantom'
            )
            plan_resource_data.ready()

        logging.info('Finished making plans!')

    def on_execute(self):
        self.make_plain_text_queries()
        self.make_plans()

    def on_release(self, params):
        super(GenerateEntityRecommenderRequests, self).on_release(params)
        self.mark_released_resources(params["release_status"], ttl=21)
