from base64 import b64encode
from base64 import urlsafe_b64decode

import itertools
import logging
import yt.wrapper as yt

from sandbox.projects.dj.services.entity.common import EXPERIMENT_MIXED
from sandbox.projects.dj.services.entity.helpers import make_file_name_for_experiment


class FilterRequestsByUriPrefixes(object):
    def __init__(self, uri_prefixes, experiments, generate_apphost_ammo):
        self.uri_prefixes = set(uri_prefixes or [])
        self.experiments = {e:i for i, e in enumerate(experiments)}
        self.generate_apphost_ammo = generate_apphost_ammo

    def __call__(self, row):
        if self.uri_prefixes and row['path'] not in self.uri_prefixes:
            return
        # If only mixed exp is specified, we don't need to filter anything.
        if not (len(self.experiments) == 1 and EXPERIMENT_MIXED in self.experiments) \
            and row['experiment'] not in self.experiments:
            return
        if EXPERIMENT_MIXED in self.experiments:
            yield yt.create_table_switch(self.experiments[EXPERIMENT_MIXED])
            yield row
        if row['experiment'] in self.experiments:
            yield yt.create_table_switch(self.experiments[row['experiment']])
            yield row


class PlainTextQueriesMaker(object):
    def __init__(self, cluster, token, file_suffix, requests_count, log_path, uri_prefixes, experiments, shuffle, generate_apphost_ammo):
        self.client = yt.YtClient(proxy=cluster, token=token)
        self.file_suffix = file_suffix
        self.requests_count = requests_count
        self.log_path = log_path
        self.uri_prefixes = uri_prefixes
        self.experiments = experiments
        self.shuffle = shuffle
        self.generate_apphost_ammo = generate_apphost_ammo
        self.output_tables = []

    def run(self):
        logging.info('Started preparing requests table...')
        self.make_tables()
        logging.info('Finished preparing requests table!')

        logging.info('Started making plain text query files...')
        self.make_files()
        logging.info('Finished making plain text query files!')

    def make_tables(self):
        self.output_tables.extend([self.client.create_temp_table() for e in self.experiments])
        mapper = FilterRequestsByUriPrefixes(self.uri_prefixes, self.experiments, self.generate_apphost_ammo)
        self.client.run_map(mapper, self.get_log_table(self.log_path), self.output_tables)
        if self.shuffle:
            with yt.OperationsTracker() as tracker:
                for output_table in self.output_tables:
                    sort_operation = self.client.run_sort(output_table, output_table, sort_by=['random'], sync=False)
                    tracker.add(sort_operation)
                tracker.wait_all()

    def make_files(self):
        for table, experiment in itertools.izip(self.output_tables, self.experiments):
            file_name = make_file_name_for_experiment(experiment, self.file_suffix)
            logging.info('Transferring data from {} to {}...'.format(table, file_name))
            with open(file_name, 'w') as output_file:
                self.make_plain_text_requests_file(table, output_file)

    def make_plain_text_requests_file(self, table, output_file):
        for i, row in enumerate(self.client.read_table(table)):
            if i >= self.requests_count:
                break
            if self.generate_apphost_ammo:
                request = row['apphost_body']
                output_file.write('{}\n'.format(request))
            else:
                request = self.make_request(row['path'], row['http_body'])
                output_file.write('{}\n{}\r\n'.format(len(request), request))
        else:
            assert i >= self.requests_count

    def get_log_table(self, log_path):
        log_path = log_path.rstrip('/')
        if self.client.get_type(log_path) == 'table':
            return log_path
        list_dir = sorted(self.client.list(log_path), reverse=True)
        if not list_dir:
            raise RuntimeError("Empty dir")
        return yt.ypath_join(log_path, list_dir[0])

    @staticmethod
    def make_request(url, post_body):
        request = 'POST {} HTTP/1.0\r\n'.format(url)
        request += 'Content-Length: {}\r\n'.format(len(post_body))
        request += 'User-Agent: Yandex.Tank (sandbox)\r\n\r\n'
        request += post_body
        return request
