from sandbox.sandboxsdk.network import is_port_free
from sandbox.sdk2.helpers import subprocess as sp

import time


class EntityRecommender:
    def __init__(
        self,
        binary_path,
        http_port,
        apphost_port,
        config_path,
        patch_paths,
        working_dir_path=None,
        realtime_update_config_path=None,
        process_log=None,
        access_log_path=None,
        verbose=0
    ):
        self.binary_path = binary_path
        self.http_port = http_port
        self.apphost_port = apphost_port
        self.config_path = config_path
        self.patch_paths = patch_paths
        self.working_dir_path = working_dir_path
        self.realtime_update_config_path = realtime_update_config_path
        self.process_log = process_log
        self.access_log_path = access_log_path
        self.verbose = verbose
        self.process = None
        self.args = None

    def run(self):
        if self.process:
            return False
        self._make_args()
        self._make_popen_args()
        self._launch_process()
        return True

    def terminate(self):
        if self.process:
            self.process.terminate()
            self.process = None

    def __enter__(self):
        self.run()
        return self

    def __exit__(self, exc_type, exc_value, exc_traceback):
        self.terminate()

    def _make_args(self):
        self.args = [self.binary_path]
        self.args.extend(self._make_required_args())
        self.args.extend(self._make_optional_args())

    def _make_popen_args(self):
        self.popen_args = {}
        self.popen_args.update(self._make_popen_required_args())
        self.popen_args.update(self._make_popen_optional_args())

    def _launch_process(self):
        self.process = sp.Popen(self.args, **self.popen_args)
        while is_port_free(self.http_port) or is_port_free(self.apphost_port):
            time.sleep(1)
            if self.process.poll() is not None:
                raise RuntimeError("Error in Entity Recommender starting")

    def _make_required_args(self):
        return [
            "--config", self.config_path,
            "--http-port", str(self.http_port),
            "--apphost-port", str(self.apphost_port),
        ]

    def _make_optional_args(self):
        args = []
        if self.working_dir_path:
            args.extend(["--cwd", self.working_dir_path])
        if self.realtime_update_config_path:
            args.extend(["--realtime-update-config", self.realtime_update_config_path])
        if self.access_log_path:
            args.extend(["--access-log", self.access_log_path])
        if self.verbose:
            args.extend(["--v", str(self.verbose)])
        for patch_path in self.patch_paths:
            args.extend(["--patch", patch_path])
        return args

    def _make_popen_required_args(self):
        return {
            "stderr": sp.STDOUT,
        }

    def _make_popen_optional_args(self):
        args = {}
        if self.process_log:
            args["stdout"] = self.process_log.stdout
        return args
