import os
import tarfile
import shutil
import tempfile
import uuid
from sandbox import sdk2
from sandbox.common.types.resource import State
from sandbox.projects.dj.services.entity.common import EXPERIMENT_MIXED
from sandbox.projects.dj.services.entity.components import EntityRecommender
from sandbox.projects.dj.services.entity.resources import EntityRecommenderShootingPlan
from sandbox.projects.dj.services.entity.resources import EntityRecommenderShard2
from sandbox.projects.dj.services.entity.resources import EntityRecommenderBundle
from sandbox.projects.dj.services.entity.resources import EntityRecommenderArcadiaModels
from sandbox.projects.dj.services.entity.resources import EntityRecommenderArcadiaDeepModel


DEFAULT_SHARD2_FILENAME = "shard2.bin"


class TemporaryDirectory(object):
    def __init__(self, dir=None):
        self.directory = tempfile.mkdtemp(dir=dir)

    def __enter__(self):
        return self.directory

    def __exit__(self, exc_type, exc_val, exc_tb):
        shutil.rmtree(self.directory, ignore_errors=True)


def make_abs_path_relative_to(basefile, targetfile):
    return os.path.join(os.path.abspath(os.path.dirname(basefile)), targetfile)


def resource_data_path(resource):
    return str(resource.path)


def init_resource(
        resource_type,
        explicit_resource=None,
        resource_id=None,
        resource_attributes=None,
        resource_status=State.READY):
    if explicit_resource:
        return explicit_resource
    if resource_id:
        return find_latest_resource(resource_type, id=resource_id, status=resource_status)
    if resource_attributes:
        return find_latest_resource(resource_type, attrs=resource_attributes, status=resource_status)
    raise RuntimeError("Can not find {type} resource".format(type=resource_type))


def find_latest_resource(*args, **kwargs):
    resource = sdk2.Resource.find(*args, **kwargs).limit(1).first()
    if not resource:
        raise RuntimeError("Can not find resource")
    return resource


def launch_entity_recommender(*args, **kwargs):
    recommender = EntityRecommender(*args, **kwargs)
    recommender.run()
    return recommender


def prepare_entity_recommender_resources(
        bundle_archive, shard2_file, models, dssm, output_dir_path):
    bundle_archive_path = resource_data_path(bundle_archive)
    shard2_file_path = resource_data_path(shard2_file)
    models_path = resource_data_path(models)
    dssm_path = resource_data_path(dssm)
    prepare_entity_recommender_bundle(bundle_archive_path, output_dir_path)
    prepare_entity_recommender_models(models_path, output_dir_path)
    prepare_entity_recommender_dssm(dssm_path, output_dir_path + '/models/dssm')
    prepare_entity_recommender_shard2(shard2_file_path, output_dir_path)


def generate_patch_path(output_dir_path):
    return os.path.join(output_dir_path, 'patch_{}.pbtxt'.format(uuid.uuid4()))


def create_patch_file(patch, output_dir_path):
    if not patch:
        return None
    patch_file_path = generate_patch_path(output_dir_path)
    with open(patch_file_path, 'w') as patch_file:
        patch_file.write(str(patch))
    return patch_file_path


def prepare_patches(bundle_patches, patch, output_dir_path):
    patch_paths = [os.path.join(output_dir_path, x) for x in bundle_patches]
    patch_from_parameters = create_patch_file(patch, output_dir_path)
    if patch_from_parameters is not None:
        patch_paths.append(patch_from_parameters)
    return patch_paths


def prepare_entity_recommender_bundle(archive_path, output_dir_path):
    with tarfile.open(archive_path, "r:gz") as tar:
        tar.extractall(output_dir_path)
        return [os.path.join(output_dir_path, filename) for filename in tar.getnames()]


def prepare_entity_recommender_models(models_path, output_dir_path):
    if not os.path.exists(output_dir_path):
        os.makedirs(output_dir_path)
    with tarfile.open(models_path, "r:gz") as tar:
        tar.extractall(output_dir_path)


def prepare_entity_recommender_dssm(resource_path, output_dir_path):
    if not os.path.exists(output_dir_path):
        os.makedirs(output_dir_path)
    with tarfile.open(resource_path, "r:gz") as tar:
        tar.extractall(output_dir_path)


def prepare_entity_recommender_shard2(shard2_path, output_dir_path):
    target_shard2_path = os.path.join(output_dir_path, DEFAULT_SHARD2_FILENAME)
    os.symlink(shard2_path, target_shard2_path)
    return target_shard2_path


def find_shard2_resource(explicit_resource):
    return init_resource(
        EntityRecommenderShard2,
        explicit_resource=explicit_resource,
        resource_attributes={"released": "stable"})


def find_bundle_resource(explicit_resource):
    return init_resource(
        EntityRecommenderBundle,
        explicit_resource=explicit_resource,
        resource_attributes={"released": "stable"})


def find_models_resource(explicit_resource):
    return init_resource(
        EntityRecommenderArcadiaModels,
        explicit_resource=explicit_resource,
        resource_attributes={"released": "stable"})


def find_dssm_resource():
    return init_resource(
        EntityRecommenderArcadiaDeepModel,
        resource_attributes={"released": "stable"})


def find_shooting_plan_resource(explicit_resource=None, plan_attributes=None, explicit_id=None):
    return init_resource(
        EntityRecommenderShootingPlan,
        explicit_resource=explicit_resource,
        resource_id=explicit_id,
        resource_attributes=plan_attributes)


def make_file_name_for_experiment(experiment, suffix, symbol='_'):
    return symbol.join([word for word in [experiment, suffix] if word])


def generate_ammo_update_experiments_list(experiments):
    if len(experiments) == 0:
        return [EXPERIMENT_MIXED]  # Generate only mixed
    elif len(experiments) == 1:
        return experiments  # Nothing to mix
    else:
        return [x for x in experiments] + [EXPERIMENT_MIXED]


class CmdArgsBuilder(object):
    def __init__(self):
        self.args = []

    def add_positional_argument(self, value):
        self.args.append(value)
        return self

    def add_argument(self, name, value):
        self.args.extend([name, value])
        return self

    def add_optional_argument(self, name, value):
        if value:
            self.add_argument(name, value)
        return self

    def add_optional_flag(self, name, value):
        if value:
            self.args.append(name)
        return self

    def add_arguments_list(self, prefix, values):
        if not values:
            return self
        for value in values:
            self.add_argument(prefix, value)
        return self

    def build(self):
        return [str(arg) or "" for arg in self.args]
