from sandbox import sdk2

from sandbox.projects.dj.unity.resources import DjUnityPackage
from sandbox.sandboxsdk import environments

import sandbox.common.types.task as ctt
import sandbox.projects.release_machine.core.task_env as rm_task_env

from acceptance import (AcceptanceState, ProcessState, PRODUCTION, RELEASE)

import copy
import datetime
import json
import logging
import os
import tarfile
import time

ALICE_LOG_DIR = "//home/logfeller/logs/megamind-proactivity-log/30min"
ANSWERS_REDIR_LOG_DIR = "//home/logfeller/logs/answ-redir-log/30min"
IMAGES_LOG_DIR = "//user_sessions/pub/images/fast"
MUSIC_USER_EVENTS_LOG_DIR = "//logs/music-user-events-log/1h"
SEARCH_LOG_DIR = "//user_sessions/pub/search/fast"
SPY_LOG_DIR = "//user_sessions/pub/spy_log_v2/fast"
VIDEO_LOG_DIR = "//user_sessions/pub/video/fast"
WATCH_LOG_DIR = "//user_sessions/pub/watch_log_tskv/fast"
TOLOKA_EVENTS_LOG_DIR = "//logs/toloka-dj-log/1h"


class DjUnityAcceptance(sdk2.Task):
    """ Task for unity releases acceptance """

    class Requirements(sdk2.Task.Requirements):
        environments = (
            rm_task_env.TaskRequirements.startrek_client,
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment("yandex-yt-yson-bindings-skynet"),
            environments.PipEnvironment('yandex-yt-transfer-manager-client')
        )
        client_tags = rm_task_env.TaskTags.startrek_client

    class Parameters(sdk2.Task.Parameters):
        package_resource = sdk2.parameters.Resource(
            "Resource with Unity package",
            resource_type=DjUnityPackage,
            required=True
        )
        nirvana_token_vault = sdk2.parameters.String(
            "Nirvana token vault",
            required=True,
            default='robot_dj_unity_nirvana_token'
        )
        component_name = sdk2.parameters.String(
            "Release machine component name for startrek ticket update",
            default='dj_unity'
        )
        release_number = sdk2.parameters.Integer(
            "Release number for startrek ticket update",
            default=None
        )
        acceptance_base_dir = sdk2.parameters.String(
            "Acceptance YT dir",
            default='//home/dj_unity/acceptance'
        )
        yt_server = sdk2.parameters.String(
            'YT server',
            required=True,
            default='arnold'
        )
        yt_token = sdk2.parameters.String(
            'YT token vault',
            required=True,
            default = 'robot_dj_unity_yt_token'
        )
        kill_timeout = 24 * 3600

    def build_base_dir(self):
        return "{}/{}/{}".format(
            self.Parameters.acceptance_base_dir,
            str(self.Parameters.release_number),
            str(self.Parameters.package_resource.id)
        )

    def build_watch_log_dir(self):
        import yt.wrapper as yt
        return yt.ypath_join(self.build_base_dir(), "input", "watch_log")

    def build_watch_log_dated_dir(self):
        import yt.wrapper as yt
        return yt.ypath_join(self.build_watch_log_dir(), self.Context.date_str)

    def build_search_log_dir(self):
        import yt.wrapper as yt
        return yt.ypath_join(self.build_base_dir(), "input", "search_log")

    def build_search_log_dated_dir(self):
        import yt.wrapper as yt
        return yt.ypath_join(self.build_search_log_dir(), self.Context.date_str)

    def build_spy_log_dir(self):
        import yt.wrapper as yt
        return yt.ypath_join(self.build_base_dir(), "input", "spy_log")

    def build_spy_log_dated_dir(self):
        import yt.wrapper as yt
        return yt.ypath_join(self.build_spy_log_dir(), self.Context.date_str)

    def build_images_log_dir(self):
        import yt.wrapper as yt
        return yt.ypath_join(self.build_base_dir(), "input", "images_log")

    def build_images_log_dated_dir(self):
        import yt.wrapper as yt
        return yt.ypath_join(self.build_images_log_dir(), self.Context.date_str)

    def build_video_log_dir(self):
        import yt.wrapper as yt
        return yt.ypath_join(self.build_base_dir(), "input", "video_log")

    def build_video_log_dated_dir(self):
        import yt.wrapper as yt
        return yt.ypath_join(self.build_video_log_dir(), self.Context.date_str)

    def build_answers_redir_log(self):
        import yt.wrapper as yt
        return yt.ypath_join(self.build_base_dir(), "input", "answers_redir_log", self.Context.date_str)

    def build_music_user_events_log(self):
        import yt.wrapper as yt
        return yt.ypath_join(self.build_base_dir(), "input", "music_user_events_log", self.Context.date_str)

    def build_alice_log_dir(self):
        import yt.wrapper as yt
        return yt.ypath_join(self.build_base_dir(), "input", "alice_log_dir")

    def build_alice_log_dated_table(self):
        import yt.wrapper as yt
        return yt.ypath_join(self.build_alice_log_dir(), self.Context.date_str)

    def build_alice_base_dir(self):
        import yt.wrapper as yt
        return yt.ypath_join(self.build_base_dir(), "alice")

    def build_toloka_events_log(self):
        import yt.wrapper as yt
        return yt.ypath_join(self.build_base_dir(), "input", "toloka_events_log", self.Context.date_str)

    def prepare_input_tables(self):
        import yt.wrapper as yt
        yt.config.set_proxy(self.Parameters.yt_server)
        yt.config['token'] = sdk2.Vault.data(self.Parameters.yt_token)

        current = datetime.datetime.now() - datetime.timedelta(days=2)
        while current < datetime.datetime.now():
            fast_dir_name = str(int(time.mktime(datetime.datetime(
                current.year,
                current.month,
                current.day,
                current.hour
            ).timetuple())))

            exists = True
            for folder in [WATCH_LOG_DIR, IMAGES_LOG_DIR, VIDEO_LOG_DIR, SPY_LOG_DIR]:
                for table_name in ("clean", "yandex_staff"):
                    table_path = yt.ypath_join(folder, fast_dir_name, table_name)
                    if not yt.exists(table_path):
                        logging.info("Can not find log table {}".format(table_path))
                        exists = False
                        fast_dir_name = None
                        current += datetime.timedelta(minutes=30)
                        break
                if not exists:
                    break

            if exists:
                break

        for dirs in (
            (WATCH_LOG_DIR, self.build_watch_log_dated_dir()),
            (IMAGES_LOG_DIR, self.build_images_log_dated_dir()),
            (VIDEO_LOG_DIR, self.build_video_log_dated_dir()),
            (SPY_LOG_DIR, self.build_spy_log_dated_dir())
        ):
            for table_name in ("clean", "yandex_staff"):
                yt.copy(
                    yt.ypath_join(dirs[0], fast_dir_name, table_name),
                    yt.ypath_join(dirs[1], table_name),
                    recursive=True
                )
            # Stub outstaff table
            yt.copy(
                yt.ypath_join(dirs[0], fast_dir_name, "yandex_staff"),
                yt.ypath_join(dirs[1], "outstaff"),
                recursive=True
            )

        # Search logs
        current = datetime.datetime.now() - datetime.timedelta(days=2)
        while current < datetime.datetime.now():
            fast_dir_name = str(int(time.mktime(datetime.datetime(
                current.year,
                current.month,
                current.day,
                current.hour
            ).timetuple())))
            exists = True
            for table_name in ("clean", "yandex_staff"):
                table_path = yt.ypath_join(SEARCH_LOG_DIR, fast_dir_name, table_name)
                if not yt.exists(table_path):
                    logging.info("Can not find log table {}".format(table_path))
                    exists = False
                    fast_dir_name = None
                    current += datetime.timedelta(minutes=30)
                    break

            if exists:
                break

        for table_name in ("clean", "yandex_staff"):
            yt.copy(
                yt.ypath_join(SEARCH_LOG_DIR, fast_dir_name, table_name),
                yt.ypath_join(self.build_search_log_dated_dir(), table_name),
                recursive=True
            )
        # Stub outstaff table
        yt.copy(
            yt.ypath_join(SEARCH_LOG_DIR, fast_dir_name, "yandex_staff"),
            yt.ypath_join(self.build_search_log_dated_dir(), "outstaff"),
            recursive=True
        )

        yt.copy(
            yt.ypath_join(ANSWERS_REDIR_LOG_DIR, sorted(yt.list(ANSWERS_REDIR_LOG_DIR))[0]),
            self.build_answers_redir_log(),
            recursive=True
        )

        yt.config.set_proxy("hahn")
        import yt.transfer_manager.client as tm
        tm_client = tm.TransferManager(token=sdk2.Vault.data(self.Parameters.yt_token))

        last_music_user_events_log = sorted(yt.list(MUSIC_USER_EVENTS_LOG_DIR))[0]
        tm_client.add_task(
            "hahn",
            yt.ypath_join(MUSIC_USER_EVENTS_LOG_DIR, last_music_user_events_log),
            self.Parameters.yt_server,
            self.build_music_user_events_log(),
            sync=True
        )

        last_alice_log = sorted(yt.list(ALICE_LOG_DIR))[-1]
        tm_client.add_task(
            "hahn",
            yt.ypath_join(ALICE_LOG_DIR, last_alice_log),
            self.Parameters.yt_server,
            self.build_alice_log_dated_table(),
            sync=True
        )

        last_toloka_events_log = sorted(yt.list(TOLOKA_EVENTS_LOG_DIR))[0]
        tm_client.add_task(
            "hahn",
            yt.ypath_join(TOLOKA_EVENTS_LOG_DIR, last_toloka_events_log),
            self.Parameters.yt_server,
            self.build_toloka_events_log(),
            sync=True
        )

    def prepare_yt_folder(self):
        import yt.wrapper as yt
        yt.config.set_proxy(self.Parameters.yt_server)
        yt.config['token'] = sdk2.Vault.data(self.Parameters.yt_token)

        base_dir = self.build_base_dir()
        if not yt.exists(base_dir):
            yt.create('map_node', base_dir, recursive=True, force=False)
            exp_time = datetime.datetime.now() + datetime.timedelta(days=3)
            yt.set_attribute(base_dir, "expiration_time", exp_time.isoformat(sep=' '))
            self.prepare_input_tables()

    @staticmethod
    def find_production_package_resource():
        logging.info('Searching for production package')
        res = sdk2.Resource.find(
            type=DjUnityPackage,
            attrs={
                "released": ctt.ReleaseStatus.STABLE
            }
        ).order(-sdk2.Resource.id).first()
        logging.info('Production package: {}'.format(res.id))
        return res

    @staticmethod
    def load_package(resource, env):
        logging.info('Unpacking {} package'.format(env))
        package_resource_data = sdk2.ResourceData(resource)
        with tarfile.open(str(package_resource_data.path), "r:gz") as tar:
            tar.extractall(path="./{}".format(env))

    @staticmethod
    def load_processes_json(env):
        logging.info('Reading {} processes file'.format(env))
        with open("./{}/configs/processes.json".format(env), 'r') as f:
            return json.load(f)

    @staticmethod
    def load_process_json(process, env):
        process_file = "./{}/{}".format(env, process)
        logging.info('Reading process file {}'.format(process_file))

        if not os.path.isfile(process_file):
            logging.info('Process file {} does not exist'.format(process_file))
            return None

        with open(process_file, 'r') as f:
            return json.load(f)

    def create_global_params(self):
        return {
            '$(TIMESTAMP_FORMATTED)': datetime.datetime.now().strftime('%Y-%m-%dT%H:%M:%S%z'),
            '$(DATE_STR)': self.Context.date_str,
            '$(WATCH_LOG_BASE_DIR)': self.build_watch_log_dir(),
            '$(WATCH_LOG_DIR)': self.build_watch_log_dated_dir(),
            '$(IMAGES_LOG_BASE_DIR)': self.build_images_log_dir(),
            '$(SEARCH_LOG_BASE_DIR)': self.build_search_log_dir(),
            '$(SEARCH_LOG_DIR)': self.build_search_log_dated_dir(),
            '$(VIDEO_LOG_BASE_DIR)': self.build_video_log_dir(),
            '$(ANSWERS_REDIR_LOG)': self.build_answers_redir_log(),
            '$(MUSIC_USER_EVENTS_LOG)': self.build_music_user_events_log(),
            '$(SPY_LOG_BASE_DIR)': self.build_spy_log_dir(),
            '$(ALICE_LOG_DIR)': self.build_alice_log_dir(),
            '$(ALICE_BASE_DIR)': self.build_alice_base_dir(),
            '$(TOLOKA_EVENTS_LOG)': self.build_toloka_events_log()
        }

    def build_base_env_dir(self, env):
        return "{}/{}".format(
            self.build_base_dir(),
            env
        )

    def build_global_params(self, release_package_resource_id, production_package_resource_id):
        logging.info('Building global params')

        global_params = {RELEASE: self.create_global_params()}
        global_params[PRODUCTION] = copy.deepcopy(global_params[RELEASE])
        global_params[RELEASE]['$(PACKAGE_RESOURCE_ID)'] = str(release_package_resource_id)
        global_params[PRODUCTION]['$(PACKAGE_RESOURCE_ID)'] = str(production_package_resource_id)
        global_params[RELEASE]['$(BASE_DIR)'] = self.build_base_env_dir(RELEASE)
        global_params[PRODUCTION]['$(BASE_DIR)'] = self.build_base_env_dir(PRODUCTION)

        logging.info('Global params: {}'.format(global_params))
        return global_params

    def build_acceptance_state(
        self,
        acceptance_state,
        release_processes_json,
        production_processes_json,
    ):
        logging.info('Build acceptance state')
        for process in release_processes_json["processes"]:
            release_process_json = self.load_process_json(process, RELEASE)
            if "acceptance" in release_process_json:
                production_process_json = self.load_process_json(process, PRODUCTION)
                if production_process_json is not None:
                    if not "acceptance" in production_process_json:
                        production_process_json = None
                acceptance_state.add(release_process_json, production_process_json)

    def on_execute(self):
        with self.memoize_stage.start_processing:
            yesterday = datetime.datetime.now() - datetime.timedelta(days=1)
            self.Context.date_str = yesterday.strftime('%Y-%m-%d')

            self.prepare_yt_folder()

            production_package_resource = self.find_production_package_resource()

            self.load_package(production_package_resource, PRODUCTION)
            self.load_package(self.Parameters.package_resource, RELEASE)

            production_processes_json = self.load_processes_json(PRODUCTION)
            release_processes_json = self.load_processes_json(RELEASE)

            global_params = self.build_global_params(
                self.Parameters.package_resource.id,
                production_package_resource.id
            )

            acceptance_state = AcceptanceState(
                {
                    RELEASE: self.Parameters.package_resource.id,
                    PRODUCTION: production_package_resource.id
                },
                self.Parameters.nirvana_token_vault,
                global_params,
                self.Parameters.component_name,
                self.Parameters.release_number,
                self.build_base_dir(),
                self.Parameters.yt_server,
                self.Parameters.yt_token
            )
            self.build_acceptance_state(
                acceptance_state,
                release_processes_json,
                production_processes_json
            )

            acceptance_state.start_processes()
            self.Context.acceptance_state = acceptance_state.to_json()

        acceptance_state = AcceptanceState.from_json(self.Context.acceptance_state)
        if acceptance_state.check_processes():
            self.Context.acceptance_state = acceptance_state.to_json()
            raise sdk2.WaitTime(10 * 60) # 10min
