from sandbox import sdk2
from sandbox.sandboxsdk import environments

from sandbox.sdk2.helpers import subprocess

from sandbox.projects import resource_types
from sandbox.projects.common import dolbilka

import sandbox.common.types.client as ctc

import logging
import json
import os
import sys
import urllib


class DjUnityProfileReaderShoot(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        yt_proxy = sdk2.parameters.String("YT server name", default="arnold", required=True)
        yt_token_vault = sdk2.parameters.String("YT token vault", required=True)

        ids_table = sdk2.parameters.String("YT table with data for requests")

        rps = sdk2.parameters.Integer("RPS", default=10)
        count = sdk2.parameters.Integer("Max requests count", default=10000)

        host = sdk2.parameters.String("Host with Unity Profile Reader")

        max_failed_requests_percent = sdk2.parameters.Integer("Max failed requests percent", default=5)

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment("yandex-yt-yson-bindings-skynet", version='0.3.32-0')
        )
        client_tags = ctc.Tag.GENERIC & ctc.Tag.Group.LINUX

    def generate_requests(self, out_file):
        yt_token = sdk2.Vault.data(self.Parameters.yt_token_vault)

        import yt.wrapper as yt
        yt.config.set_proxy(self.Parameters.yt_proxy)
        yt.config['token'] = yt_token

        progress = 0
        requests_content = ''
        for row in yt.read_table(self.Parameters.ids_table):
            requests_content += "/api/dj_unity_profile_reader/get_profile?merge_different_keys=true&profile_key_list="

            ids = []
            ids.append({
                "ObjectNamespace": row["ObjectNamespace"],
                "ObjectType": row["ObjectType"],
                "ObjectId": row["ObjectId"]
            })

            requests_content += urllib.quote(json.dumps(ids))

            requests_content += '\n'
            progress += 1
            if progress % 100 == 0:
                logging.info(progress)
            if progress == self.Parameters.count:
                break

        with open(out_file, 'w') as f:
            f.write(requests_content)

    def shoot(self, plan_file):
        d_executor = dolbilka.DolbilkaExecutor()
        d_executor.mode = dolbilka.DolbilkaExecutorMode.PLAN_MODE
        d_executor.max_simultaneous_requests = self.Parameters.rps
        d_executor.mode = dolbilka.DolbilkaExecutorMode.FINGER_MODE
        d_executor.requests = self.Parameters.count

        dump_path = "./results.dump"
        d_executor.run_session(plan_file,
                               dump_path,
                               port=80,
                               host=self.Parameters.host)
        parsed_stat = d_executor.run_dumper(dump_path, "Test results")

        service_unavailable = 0 if parsed_stat["service_unavailable"] is None else\
            int(parsed_stat["service_unavailable"])

        error_requests = 0 if parsed_stat["error_requests"] is None else int(parsed_stat["error_requests"])

        if error_requests + service_unavailable > int(parsed_stat["requests"]) * self.Parameters.max_failed_requests_percent / 100:
            raise RuntimeError("Too many errors in request processing.")

    def on_execute(self):
        plain_requests_file = "./plain_requests.txt"
        self.generate_requests(plain_requests_file)

        plan_file = "./requests.plan"
        dolbilka_planner = dolbilka.DolbilkaPlanner()
        dolbilka_planner.create_plan(
            plain_requests_file,
            plan_file,
            rps=self.Parameters.rps,
            port=80,
            host=self.Parameters.host
        )

        self.shoot(plan_file)
