from sandbox import sdk2

from sandbox.projects.dj.unity.resources import DjUnityPackage
from sandbox.projects.dj.unity.utils.hitman import HitmanClient
from sandbox.projects.dj.unity.utils import nirvana

import json
import logging
import tarfile


class DjUnityReleaseProcesses(sdk2.Task):
    """ Task for releasing unity processes artifacts (configs, processes) on frozen code """

    class Parameters(sdk2.Task.Parameters):
        package_resource = sdk2.parameters.Resource(
            "Resource with Unity package",
            resource_type=DjUnityPackage,
            required=True
        )
        nirvana_token_vault = sdk2.parameters.String(
            'Nirvana token vault',
            required=True,
            default='robot_dj_unity_nirvana_token'
        )
        hitman_token_vault = sdk2.parameters.String(
            'Hitman token vault',
            required=True,
            default='robot_dj_unity_hitman_token'
        )

    def release_process(self, process, resource_id, branch, nirvana_client, hitman_client):
        logging.info('Creating new workflow instance')
        new_instance_id = nirvana_client.clone_workflow_instance(
            process["nirvana"]["workflow_id"],
            process["nirvana"]["instance_id"]
        )
        nirvana_client.add_comment_to_workflow_instance(
            new_instance_id,
            'Released from branch {}'.format(branch)
        )
        logging.info('New instance: {}/{}'.format(process["nirvana"]["workflow_id"], new_instance_id))
        logging.info('Setting new package resource id')
        nirvana_client.set_workflow_parameters(
            process["nirvana"]["workflow_id"],
            new_instance_id,
            [{
                "parameter": "package_resource_id",
                "value": str(resource_id)
            }]
        )
        nirvana_client.approve_workflow_instance(
            process["nirvana"]["workflow_id"],
            new_instance_id
        )
        logging.info('Updating hitman process {}'.format(process["hitman"]["process_id"]))
        hitman_client.update_workflow_id(
            process["hitman"]["process_id"],
            "{}/{}".format(process["nirvana"]["workflow_id"], new_instance_id)
        )

    def on_execute(self):
        logging.info('Creating nirvana client')
        nirvana_token = sdk2.Vault.data(self.Parameters.nirvana_token_vault)
        nirvana_client = nirvana.NirvanaClient(nirvana_token)

        logging.info('Creating hitman client')
        hitman_token = sdk2.Vault.data(self.Parameters.hitman_token_vault)
        hitman_client = HitmanClient(hitman_token)

        logging.info('Unpacking package')
        package_resource_data = sdk2.ResourceData(self.Parameters.package_resource)
        with tarfile.open(str(package_resource_data.path), "r:gz") as tar:
            tar.extractall(path=".")

        logging.info('Reading processes file')
        with open("./configs/processes.json", 'r') as f:
            processes_json = json.load(f)

        for process in processes_json["processes"]:
            logging.info('Reading process file {}'.format(process))
            with open(process, 'r') as f:
                process_json = json.load(f)
                self.release_process(
                    process_json,
                    self.Parameters.package_resource.id,
                    self.Parameters.package_resource.branch,
                    nirvana_client,
                    hitman_client
                )
