# -*- coding: utf-8 -*-

import json
import time
import logging
import requests
from urlparse import urljoin

from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry


class NirvanaException(Exception):
    pass


class NirvanaClient(object):
    def __init__(self, oauth_token):
        self.url = 'https://nirvana.yandex-team.ru/api/public/v1/'
        self.oauth_token = oauth_token
        self.session = requests.Session()
        self.session.headers['Authorization'] = 'OAuth {}'.format(self.oauth_token)
        self.session.headers['Content-Type'] = 'application/json'
        retry = Retry(
            total=10,
            backoff_factor=0.3,
        )
        adapter = HTTPAdapter(max_retries=retry)
        self.session.mount('http://', adapter)
        self.session.mount('https://', adapter)

    def make_request(self, url):
        response = self.session.get(url, verify=False)
        response.raise_for_status()
        response_content = response.json()
        logging.info('Result: {}'.format(response_content))
        return response_content['result']

    def clone_workflow_instance(self, workflow_id, instance_id):
        uri_template = ('cloneWorkflowInstance?workflowId={workflow_id}&'
                   'workflowInstanceId={instance_id}')
        uri = uri_template.format(workflow_id=workflow_id,
                             instance_id=instance_id)
        url = urljoin(self.url, uri)
        logging.info('Trying to clone workflow instance: {}/{}'.format(workflow_id, instance_id))
        try:
            return self.make_request(url)
        except (requests.HTTPError, requests.ConnectionError) as err:
            msg = 'Could not clone workflow instance {}/{}\n Details: {}'.format(workflow_id, instance_id, err)
            raise NirvanaException(msg)

    def approve_workflow_instance(self, workflow_id, instance_id):
        uri_template = ('approveWorkflow?workflowId={workflow_id}&'
                   'workflowInstanceId={instance_id}')
        uri = uri_template.format(workflow_id=workflow_id,
                             instance_id=instance_id)
        url = urljoin(self.url, uri)
        logging.info('Trying to approve workflow instance: {}/{}'.format(workflow_id, instance_id))
        try:
            return self.make_request(url)
        except (requests.HTTPError, requests.ConnectionError) as err:
            msg = 'Could not approve workflow instance {}/{}\n Details: {}'.format(workflow_id, instance_id, err)
            raise NirvanaException(msg)

    def start_workflow_instance(self, workflow_id, instance_id):
        uri_template = ('startWorkflow?workflowId={workflow_id}&'
                   'workflowInstanceId={instance_id}')
        uri = uri_template.format(workflow_id=workflow_id,
                             instance_id=instance_id)
        url = urljoin(self.url, uri)
        logging.info('Trying to start workflow instance: {}/{}'.format(workflow_id, instance_id))
        try:
            return self.make_request(url)
        except (requests.HTTPError, requests.ConnectionError) as err:
            msg = 'Could not start workflow instance {}/{}\n Details: {}'.format(workflow_id, instance_id, err)
            raise NirvanaException(msg)

    def get_workflow_instance_state(self, workflow_id, instance_id):
        uri_template = ('getExecutionState?workflowId={workflow_id}&'
                   'workflowInstanceId={instance_id}')
        uri = uri_template.format(workflow_id=workflow_id,
                             instance_id=instance_id)
        url = urljoin(self.url, uri)
        logging.info('Trying to get workflow instance state: {}/{}'.format(workflow_id, instance_id))
        try:
            return self.make_request(url)
        except (requests.HTTPError, requests.ConnectionError) as err:
            msg = 'Could not get workflow instance state {}/{}\n Details: {}'.format(workflow_id, instance_id, err)
            raise NirvanaException(msg)

    def add_comment_to_workflow_instance(self, instance_id, comment):
        uri_template = 'addCommentToWorkflowInstance?workflowInstanceId={instance_id}&comment={comment}'
        uri = uri_template.format(instance_id=instance_id, comment=comment)
        url = urljoin(self.url, uri)
        logging.info('Trying to add workflow instance comment: {}'.format(instance_id))
        try:
            return self.make_request(url)
        except (requests.HTTPError, requests.ConnectionError) as err:
            msg = 'Could not add workflow instance comment {}\n Details: {}'.format(instance_id, err)
            raise NirvanaException(msg)

    def set_workflow_parameters(self, workflow_id, instance_id, parameters_list):
        uri = 'setGlobalParameters'
        url = urljoin(self.url, uri)
        data = {"jsonrpc": "2.0",
                "method": "setGlobalParameters",
                "params": {
                    'params': parameters_list,
                    'workflowId': workflow_id,
                    'workflowInstanceId': instance_id
                },
                "id": "set_params_{}".format(workflow_id)}
        try:
            response = self.session.post(url,
                                         data=json.dumps(data),
                                         verify=False).json()
            print(response)
            logging.info('Result: {}'.format(response))
            result = response['result']
            return result
        except (requests.HTTPError, requests.ConnectionError) as err:
            msg = 'Could not set parameters for workflow instance {}/{}\n Details: {}'.format(workflow_id, instance_id, err)
            raise NirvanaException(msg)

    def remove_blocks(self, workflow_id, instance_id, blocks):
        uri = 'removeBlocks'
        url = urljoin(self.url, uri)
        data = {
                "jsonrpc": "2.0",
                "method": "removeBlocks",
                "params": {
                    'blocks': blocks,
                    'workflowId': workflow_id,
                    'workflowInstanceId': instance_id
                },
                "id": "set_blocks_{}".format(workflow_id)
        }
        try:
            response = self.session.post(url,
                                         data=json.dumps(data),
                                         verify=False).json()
            print(response)
            logging.info('Result: {}'.format(response))
            result = response['result']
            return result
        except (requests.HTTPError, requests.ConnectionError) as err:
            msg = 'Could not remove blocks for workflow instance {}/{}\n Details: {}'.format(workflow_id, instance_id, err)
            raise NirvanaException(msg)

    def get_blocks_result(self, workflow_id, instance_id, blocks):
        uri = 'getBlockResults'
        url = urljoin(self.url, uri)
        data = {
                "jsonrpc": "2.0",
                "method": "getBlockResults",
                "params": {
                    'blocks': blocks,
                    'workflowId': workflow_id,
                    'workflowInstanceId': instance_id
                },
                "id": "get_blocks_result{}".format(workflow_id)
        }
        try:
            response = self.session.post(url,
                                         data=json.dumps(data),
                                         verify=False).json()
            print(response)
            logging.info('Result: {}'.format(response))
            result = response['result']
            return result
        except (requests.HTTPError, requests.ConnectionError) as err:
            msg = 'Could not get blocks for workflow instance {}/{}\n Details: {}'.format(workflow_id, instance_id, err)
            raise NirvanaException(msg)
