import logging
# from datetime import datetime, timedelta

from sandbox import sdk2
# from sandbox.common.telegram import TelegramBot
from sandbox.sandboxsdk import environments


class MyFirstSandboxTask(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group("Checkouter database settings", collapse=True) as ch_db_settings:
            ch_db_host = sdk2.parameters.String('Checkouter database host', required=True)
            ch_db_port = sdk2.parameters.String('Checkouter database port', required=True)
            ch_db_name = sdk2.parameters.String('Checkouter database name', required=True)
            ch_db_user_secret = sdk2.parameters.YavSecret(
                'Checkouter database username'
                '\nYAV secret identifier (with optional version)'
                '\nDefault key: username',
                required=True
            )
            ch_db_password_secret = sdk2.parameters.YavSecret(
                'Checkouter database password '
                '\nYAV secret identifier (with optional version)'
                '\nDefault key: password',
                required=True
            )

        with sdk2.parameters.Group("Startrek settings", collapse=True) as st_settings:
            st_user_agent = sdk2.parameters.String('Startrack user agent', required=True)
            st_url = sdk2.parameters.String('Startrack url', required=True)
            st_token_secret = sdk2.parameters.YavSecret(
                'Startrack token'
                '\nYAV secret identifier (with optional version)'
                '\nDefault key: token',
                required=True
            )
            st_queue = sdk2.parameters.String('Startrack queue', required=True)
            st_summary = sdk2.parameters.String('Startrack ticket summary', required=True)
            st_tags = sdk2.parameters.List('List of tags for Startrek issue')
            board = sdk2.parameters.String('Board URL')

    class Requirements(sdk2.Requirements):
        disk_space = 1024 * 5
        environments = (
            environments.PipEnvironment('psycopg2-binary'),
            # environments.PipEnvironment('pymssql==2.1.4'),
            environments.PipEnvironment('yandex_tracker_client', version="1.3", custom_parameters=["--upgrade-strategy only-if-needed"]),
            environments.PipEnvironment('startrek_client', version="2.3.0", custom_parameters=["--upgrade-strategy only-if-needed"]),
        )

    def request_ch(self, user, password):
        try:
            logging.info('Connecting to the Checkouter database...')
            logging.info('User: {}'.format(user))
            import psycopg2
            with psycopg2.connect(
                host=self.Parameters.ch_db_host,
                port=self.Parameters.ch_db_port,
                database=self.Parameters.ch_db_name,
                user=user,
                password=password,
                sslmode='require',
            ) as conn:
                logging.info('Connected')

                with conn.cursor() as cur:

                    cur.execute(
                        """
                            select
                                id,
                                created_at
                            from orders
                            limit 10;
                        """
                    )

                    data = cur.fetchall()
        except Exception as error:
            logging.info(error)
        finally:
            logging.info('Checkouter database connection closed.')
        return data

    def create_startrek_ticket(self, summary, description, st_token):
        from startrek_client import Startrek
        st = Startrek(
            useragent=self.Parameters.st_user_agent,
            base_url=self.Parameters.st_url,
            token=st_token
        )

        st.issues.create(
            queue=self.Parameters.st_queue,
            summary=self.Parameters.st_summary + ' - ' + summary,
            type={'name': 'Technical Task'},
            description=description,
            tags=self.Parameters.st_tags
        )

    def on_execute(self):
        ch_db_user = self.Parameters.ch_db_user_secret.data()
        ch_db_user_key = self.Parameters.ch_db_user_secret.default_key or 'username'
        ch_db_user = ch_db_user[ch_db_user_key]
        ch_db_password = self.Parameters.ch_db_password_secret.data()
        ch_db_password_key = self.Parameters.ch_db_password_secret.default_key or 'password'
        ch_db_password = ch_db_password[ch_db_password_key]
        st_token = self.Parameters.st_token_secret.data()
        st_token_key = self.Parameters.st_token_secret.default_key or 'token'
        st_token = st_token[st_token_key]
        
        data = self.request_ch(ch_db_user, ch_db_password)
        ticket_description = '| order\\_id | created\\_at |\n| ---: | :--- |'
        for order_id, created_at in data:
            ticket_description = ticket_description + '\n| {} | {} |'.format(order_id, created_at)
        self.create_startrek_ticket('Test ticket', ticket_description, st_token)
