import logging
import os
import tarfile
import tempfile

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.projects.ydo import (
    execute_cmd,
)
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.ydo.parameters_view import generate_view, ResourceWrapper

from sandbox.projects.douber.DouberBuild.resource_types import DouberManageBinary, DouberManageConfig


class DouberCopyEvacuatorOrdersToYt(sdk2.Task):
    """Task copies evacuator_orders to YT"""

    class Parameters(generate_view(
        processor=DouberManageBinary,
        douber_config=ResourceWrapper(DouberManageConfig),
        check_owner=True,
    )):
        douber_production_secrets = sdk2.parameters.YavSecret("YAV douber_production_secrets secret id", required=True)
        ydo_oauth_tokens = sdk2.parameters.YavSecret("YAV YDO_OAUTH_TOKENS secret id", required=True)

        yt_folder = sdk2.parameters.String('Folder in YT', required=True)

    class Requirements(sdk2.Requirements):
        cores = 1
        dns = ctm.DnsType.DNS64
        environments = (
            PipEnvironment('requests'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        logging.info('Starting executing copy evacuator_orders to YT')

        douber_config_dir = tempfile.mkdtemp()
        tarfile.open(self.Parameters.douber_config_path).extractall(douber_config_dir)

        env = os.environ.copy()
        env['CONFIG_DIR'] = douber_config_dir
        env['YT_TOKEN'] = self.Parameters.ydo_oauth_tokens.data()['YT_TOKEN']
        env['TVM_SECRET'] = self.Parameters.douber_production_secrets.data()['tvm_secret']
        env['CAR_TAXI_KEY'] = self.Parameters.douber_production_secrets.data()['car_taxi_key']
        env['FEEDBACK_AUTH_TOKEN'] = self.Parameters.douber_production_secrets.data()['feedback_auth_token']
        env['DJANGO_SECRET_KEY'] = self.Parameters.douber_production_secrets.data()['django_secret']
        env['DB_PASSWORD'] = self.Parameters.douber_production_secrets.data()['database_password']
        env['REDIS_PASSWORD'] = self.Parameters.douber_production_secrets.data()['redis_password']

        execute_cmd(
            [
                str(self.Parameters.processor_path),
                'copy_evacuator_orders_to_yt',
                '--yt_folder', str(self.Parameters.yt_folder),
            ],
            'copy_evacuator_orders_to_yt',
            'Failed to copy evacuator_orders to YT',
            env=env,
        )

        logging.info('Done')
