# -*- coding: utf-8 -*-
import copy
import logging
from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import task as task_types
from sandbox.projects.common.build import parameters as build_parameters, YaPackage
from sandbox.projects.common.build.tasks import YaMakeTemplate
from sandbox.projects.common.nanny import nanny

from sandbox.projects.douber.DouberBuild.resource_types import (
    DouberConfig,
    DouberAppHostServiceConfig,
    DouberBinary,
    DouberManageBinary,
    DouberAppHostServiceBinary,
    DouberSettingsJson,
)


PRODUCTION = 'production'
TESTING = 'testing'

API = 'api'
ADMIN = 'admin'

CONFIG_TAR_GZ_NAME = 'config.tar.gz'
APPHOST_SERVICE_CONFIG_TAR_GZ_NAME = 'apphost_service_config.tar.gz'


class DouberBuild(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
        Douber Project build and release metatask
    """

    class Parameters(sdk2.Parameters):
        description = 'Build Douber binaries and configs'
        with sdk2.parameters.Group('Build resources:'):
            douber_binary = sdk2.parameters.Bool('Douber binary')
            manage_binary = sdk2.parameters.Bool('Manage binary')
            apphost_service_binary = sdk2.parameters.Bool('AppHost service binary')
            config_archive = sdk2.parameters.Bool('Config archive')
            apphost_service_config_archive = sdk2.parameters.Bool('AppHost service config archive')
            douber_settings_json = sdk2.parameters.Bool('Douber settings json')

        with sdk2.parameters.Group('Environment configuration:'):
            environment = sdk2.parameters.String(
                'Environment',
                description='Environment for which the resources are intended.',
                choices=(('Production', PRODUCTION), ('Testing', TESTING)),
                default=TESTING,
            )
            nanny_service = sdk2.parameters.String(
                'Nanny service',
                description='Nanny service for which the resources are intended.',
                choices=(('api', API), ('admin', ADMIN)),
                default=API
            )

        with sdk2.parameters.Group('Subtask parameters'):
            arcadia_url = build_parameters.ArcadiaUrl()
            arcadia_patch = build_parameters.ArcadiaPatch()
            strip_binaries = sdk2.parameters.Bool('Strip binaries')
            timeout = sdk2.parameters.Integer(
                'Tasks wait timeout',
                description='How much time (in seconds) to wait for subtasks to finish.',
                default=1800
            )

    class Context(sdk2.Context):
        done_enqueueing_tasks = False

    @staticmethod
    def get_build_type(environment):
        if environment == PRODUCTION:
            return 'release'
        return 'debug'

    def spawn_build_subtasks(self):
        common_task_parameters = dict(
            notifications=self.Parameters.notifications,
            arcadia_patch=self.Parameters.arcadia_patch,
            strip_binaries=self.Parameters.strip_binaries,
            build_type=self.get_build_type(self.Parameters.environment),
        )
        common_config_task_parameters = {
            'notifications': self.Parameters.notifications,
            build_parameters.ArcadiaUrl.name: self.Parameters.arcadia_url,
            build_parameters.ArcadiaPatch.name: self.Parameters.arcadia_patch,
            YaPackage.PackageTypeParameter.name: YaPackage.TARBALL,
            YaPackage.UseNewFormatParameter.name: True,
        }

        YaPackageTask = sdk2.Task['YA_PACKAGE']
        YaMakeReleaseToNannyTask = sdk2.Task['YA_MAKE_RELEASE_TO_NANNY']
        subtasks = []
        if self.Parameters.douber_binary:
            subtasks.append(BuildDouberBinary(self, description='Build douber binary ({})'.format(self.Parameters.environment), **common_task_parameters))
        if self.Parameters.manage_binary:
            subtasks.append(BuildDouberManageBinary(self, description='Build douber manage binary ({})'.format(self.Parameters.environment), **common_task_parameters))
        if self.Parameters.apphost_service_binary:
            subtasks.append(BuildDouberAppHostServiceBinary(self, description='Build douber apphost source binary ({})'.format(self.Parameters.environment), **common_task_parameters))
        if self.Parameters.config_archive:
            config_parameters = copy.deepcopy(common_config_task_parameters)
            config_parameters.update({
                YaPackage.PackagesParameter.name: 'douber/backend/config/{environment}/{service}_package.json'.format(
                    environment=self.Parameters.environment,
                    service=self.Parameters.nanny_service,
                ),
                'description': 'Build douber config archive ({}, {})'.format(self.Parameters.environment, self.Parameters.nanny_service),
                YaPackage.ResourceTypeParameter.name: DouberConfig.name,
            })
            subtasks.append(YaPackageTask(self, create_sub_task=True, **config_parameters))
        if self.Parameters.apphost_service_config_archive:
            apphost_service_config_parameters = copy.deepcopy(common_config_task_parameters)
            apphost_service_config_parameters.update({
                YaPackage.PackagesParameter.name: 'douber/apphost_service/config/{environment}/common_package.json'.format(
                    environment=self.Parameters.environment,
                ),
                'description': 'Build douber apphost service config archive ({}, {})'.format(self.Parameters.environment, self.Parameters.nanny_service),
                YaPackage.ResourceTypeParameter.name: DouberAppHostServiceConfig.name,
            })
            subtasks.append(YaPackageTask(self, create_sub_task=True, **apphost_service_config_parameters))
        if self.Parameters.douber_settings_json:
            subtasks.append(YaMakeReleaseToNannyTask(
                self,
                create_sub_task=True,
                description='Build douber settings json',
                result_single_file=True,
                arts_source=DouberSettingsJson.arcadia_build_path + '/douber_settings.json',
                result_rd='Douber settings json',
                test=True,
                result_rt=DouberSettingsJson.name,
                use_aapi_fuse=True,
                build_system='semi_distbuild',
                **common_task_parameters
            ))

        for task in subtasks:
            task.enqueue()
        self.Context.done_enqueueing_tasks = True
        raise sdk2.WaitTask(subtasks, task_types.Status.Group.FINISH, wait_all=True, timeout=self.Parameters.timeout)

    def on_execute(self):
        if not self.Context.done_enqueueing_tasks:
            self.spawn_build_subtasks()
        else:
            for task in self.find():
                logging.debug(task.id)
                if task.status != task_types.Status.SUCCESS:
                    raise errors.TaskFailure('One of child tasks is not in status SUCCESS!')

    def on_release(self, parameters):
        parameters['release_comments'] = 'Release to {} {}'.format(self.Parameters.environment, self.Parameters.nanny_service)
        logging.debug('Release params: %s', str(parameters))
        for task in self.find():
            logging.debug(task.id)
            task.on_release(parameters)
        nanny.ReleaseToNannyTask2.on_release(self, parameters)
        sdk2.Task.on_release(self, parameters)


class BuildDouberBinary(YaMakeTemplate.YaMakeTemplate):
    """
        Build Douber binary.
    """
    class Parameters(YaMakeTemplate.get_project_params([DouberBinary])):
        pass


class BuildDouberManageBinary(YaMakeTemplate.YaMakeTemplate):
    """
        Build Douber manage binary.
    """
    class Parameters(YaMakeTemplate.get_project_params([DouberManageBinary])):
        pass


class BuildDouberAppHostServiceBinary(YaMakeTemplate.YaMakeTemplate):
    """
        Build Douber AppHost service binary.
    """
    class Parameters(YaMakeTemplate.get_project_params([DouberAppHostServiceBinary])):
        pass
