import logging
import json
from sandbox.projects.dsearch.common.context import ReclusterCtx
from sandbox import common


class DeclusterPreparePhase1(ReclusterCtx):
    """
    How to run from console:

    msearch-trusty-dev:~/dev_sand/sandbox/projects/dsearch/DiskDecluster $
    cd ~/dev_sand/sandbox
    source ~/dev_sand/runtime_data/venv/bin/activate
    ~/dev_sand/runtime_data/venv/bin/python2.7

    import logging
    import sys
    from copy import deepcopy
    logging.basicConfig(stream=sys.stdout, level=logging.DEBUG)
    logging.info("Recluster revision")

    from yasmapi import GolovanRequest
    from projects.dsearch.DiskDecluster.steps import DeclusterPreparePhase1
    from projects.dsearch.DiskDecluster.steps import DeclusterPreparePhase2
    from projects.dsearch.DiskDecluster.steps import DeclusterPreparePhase3

    import json, time
    import requests
    from requests.packages.urllib3.exceptions import InsecureRequestWarning

    requests.packages.urllib3.disable_warnings(InsecureRequestWarning)

    oauth_token="NANNY_TOKEN_HERE"
    Phase1 = DeclusterPreparePhase1(oauth_token=oauth_token, yasmapi=GolovanRequest, dryrun = True)
    last_revision = Phase1.run()
    Phase2 = DeclusterPreparePhase2(oauth_token = oauth_token, mage_last_rev = last_revision, dryrun = True)
    snapshots_for_activate=Phase2.run()
    Phase3=DeclusterPreparePhase3(oauth_token = oauth_token, mage_last_rev = last_revision, snapshots_for_activate=snapshots_for_activate, dryrun = True)
    decluster_status=Phase3.run()
    """

    def __init__(self, oauth_token=False, yasmapi=None, dryrun=True, force=False):
        super(DeclusterPreparePhase1, self).__init__()
        logging.info("ReclusterPreparePhase2 prepare initialization")
        self.oauth_token = oauth_token
        self.yasmapi = yasmapi
        self.dryrun = dryrun
        self.force = force

    def run(self):
        # Some checks here
        # 0. Mage checks - check that decluster in mage in finished state
        # 1. yasm checks (5xx for kamaji < 5rps, copied percentage > 99, copy rate < 100mb) for all groups
        # 2. Check that this revision exists on nanny cluster
        # 3. serchmap check? that all prev instances existing in new decluster map (actually this check need us in recluster task)

        # returned revision from mage?

        logging.info("Decluster Phase 1 is starting.")

        mage_last_rev_status = self.get_mage_last_revision(self.mage_url)

        # 1. Mage check
        if not mage_last_rev_status and not self.dryrun:
            raise common.errors.TaskFailure("Mage ruchkas return error. Exiting. Details: {0}".format(str(mage_last_rev_status)))
        elif self.dryrun:
            logging.info('Dryrun fake. Mage ruchkas return error. Exiting. Details: {0}'.format(str(mage_last_rev_status)))

        logging.info('Mage get last rev: {0}'.format(str(mage_last_rev_status)))

        if mage_last_rev_status['status'] != 'recluster_checked':
            if not self.dryrun:
                raise common.errors.TaskFailure("Previous reclusterization in {0} state. "
                                                "But we need 'recluster_checked' state if we want continue. "
                                                "Exiting.".format(
                    str(mage_last_rev_status)))
            else:
                logging.info("Lastrev status mage return {0}, but dryrun reached. Continue.".format(str(mage_last_rev_status)))
                checked_last_rev = mage_last_rev_status['revision']
        else:
            logging.info("Mage last rev check passed. Status:{0}".format(str(mage_last_rev_status)))
            checked_last_rev = mage_last_rev_status['revision']

        # 2. Yasm check
        check_yasm_signals_prestable = self.yasm_check("prestable")
        if not check_yasm_signals_prestable:
            if not self.dryrun and not self.force:
                raise common.errors.TaskFailure("Yasm prestable checks return False")
            else:
                logging.info("Yasm prestable checks return False, but dryrun or force option reached. Continue.")
        else:
            logging.info("Yasm prestable checks passed.")

        check_yasm_signals_prod = self.yasm_check("prod")
        if not check_yasm_signals_prod:
            if not self.dryrun and not self.force:
                raise common.errors.TaskFailure("Yasm prod checks return False")
            else:
                logging.info("Yasm prod checks return False, but dryrun or force option reached. Continue.")
        else:
            logging.info("Yasm prod checks passed.")

        # 3. Nanny releases check

        for group in self.nanny_groups:
            # check for every group separately
            cur_releases = self.get_gencfg_releases([group])
            rev_flag = False
            for rev in cur_releases[group]:
                if rev.split("/")[1] == checked_last_rev:
                    rev_flag = True
            if not rev_flag:
                raise common.errors.TaskFailure("Rev for decluster: {0} does not finded in nanny group: {1}".format(checked_last_rev, group))
            else:
                logging.info("Nanny releases check passed for group: {0}".format(group))

        # 4. Nanny state check
        if not self.check_nanny_active(self.nanny_groups):
            raise common.errors.TaskFailure("Cannot continue. "
                                            "Last revision in groups: {0} must be in ACTIVE or ACTIVATING state."
                                            .format(self.nanny_groups))

        gencfg_nanny_releases = self.get_gencfg_releases(self.nanny_groups)
        nanny_uniq_release = self.check_gencfg_nanny_releases(gencfg_nanny_releases)

        # if there one release for nanny groups we must exit witj False

        if nanny_uniq_release:
            raise common.errors.TaskFailure("ERROR! Only one nanny rev found. "
                                            "We must add recluster revision before we can continue. "
                                            "Aborting. {0}".format(gencfg_nanny_releases))
        else:
            logging.info("Nanny uniq release check passed.")

        # set decluster status only if dryrun not specified
        if not self.dryrun:
            logging.info("Phase 1 Setting mage_last revision to 'decluster'")
            # TODO: set only status - we dont need revision here
            mage_set_last_rev_status = self.set_mage_last_revision(self.mage_url,
                                                                   checked_last_rev,
                                                                   'decluster')
        else:
            logging.info("Dryrun fake. Phase 1 setting mage_last revision to 'decluster'")
            mage_set_last_rev_status = "dryrun"

        if not mage_set_last_rev_status:
            raise common.errors.TaskFailure("Cannot set mage rev status in decluster phase1. "
                                            "Exiting. Details:{0}".format(mage_set_last_rev_status))

        return checked_last_rev


class DeclusterPreparePhase2(ReclusterCtx):

    def __init__(self, oauth_token=False, mage_last_rev=False, dryrun=True):
        logging.info("DeclusterPreparePhase2 prepare initialization")
        super(DeclusterPreparePhase2, self).__init__()
        self.oauth_token = oauth_token
        self.mage_last_rev = mage_last_rev
        self.dryrun = dryrun

    def run(self):

        logging.info("Recluster Phase 2 is starting.")

        if not self.mage_last_rev:
            raise common.errors.TaskFailure("Gencfg last revision for decluster is not specified. Aborting.")

        update_nanny_files_url = "{nanny_url}services/{servicename}/runtime_attrs/resources/files/static_files/"

        decluster_rev = "{0}:tags/{1}".format(self.searchmap_prefix,
                                              self.mage_last_rev)
        logging.info("Decluster revision is: {0}".format(decluster_rev))

        decluster_steps_ph2 = [{"comment": "Sandbox auto task update {revision_filename}".format(
            revision_filename=self.revision_filename),
                                "url": update_nanny_files_url,
                                "meta": {},
                                "oauth_token": self.oauth_token,
                                "put":
                                    {
                                        "content": {"local_path": self.revision_filename,
                                                    "content": decluster_rev,
                                                    "is_dynamic": True,
                                                    "extract_path": ""},
                                        "comment": "Sandbox auto decluster task. Update {revision_filename} to: {decluster_revison}".format(
                                            revision_filename=self.revision_filename,
                                            decluster_revison=decluster_rev,
                                        ),
                                        "meta_info": {"is_disposable": False}
                                    }
                                }]

        # Update self.revision_filename

        try:
            for service in self.nanny_groups:
                for step in decluster_steps_ph2:
                    logging.info("Starting step:{0}".format(step['comment']))
                    oauth_token = None
                    if step.get("oauth_token", False):
                        oauth_token = step.get("oauth_token")
                    # Do not do nothing if dryrun - only logging
                    # TODO: move dryrun logic into make_auth_req function
                    if not self.dryrun:
                        logging.info("Url: {0}, Action: put, Content: {1}".format(
                            step["url"].format(nanny_url=self.nanny_url,
                                               servicename=service),
                            step.get("put", False)))
                        ret = self.make_auth_req(step["url"].format(nanny_url=self.nanny_url,
                                                                    servicename=service),
                                                 token=oauth_token,
                                                 put=step.get("put", False),
                                                 timeout=900)
                    else:
                        logging.info("Dryrun fake. Url: {0}, Action: put, Content: {1}".format(
                            step["url"].format(nanny_url=self.nanny_url,
                                               servicename=service),
                            step.get("put", False)))
                        ret = True

                    if not ret:
                        raise common.errors.TaskFailure("Nanny request failed. Update file to decluster rev failed.")
                    logging.info("Step:{0} completed.".format(step['comment']))

        except Exception as e:
            raise common.errors.TaskFailure("Cannot decluster hosts. Aborting. Error:{0}".format(e))

        gencfg_service_groups = self.get_gencfg_releases_full(self.nanny_groups)
        update_nanny_instances_url = "{nanny_url}services/{servicename}/runtime_attrs/instances/"

        # So here we need update group with reclsuter revision, for any service and then push it.

        gencfg_service_groups_decluster = self.remove_smallest_nanny_revision(gencfg_service_groups, self.mage_last_rev)

        if not gencfg_service_groups_decluster:
            raise common.errors.TaskFailure("We cannot add revision to specified gencfg_groups. "
                                            "The group with the same revision already exists?"
                                            "Revision:{0}".format(self.mage_last_rev))

        # We update groups after self.revision_filename updated so we have all needed changes here for save last snapshot,
        # Activate this snapshot it later.
        snapshots_for_activate = {}
        try:
            logging.info("Step 2.1 - update gencfg instances for nanny groups started. Revision:{0}".format(
                self.mage_last_rev))
            for service in self.nanny_groups:
                logging.info("Update gencfg groups for: {0}".format(service))
                # remove old snapshot id
                gencfg_service_groups_decluster[service].pop('snapshot_id', None)

                gencfg_service_groups_decluster[service]['comment'] = "Sandbox auto decluster task. Remove revision for gencfg groups to: {0}".format(self.mage_last_rev)
                # Do not do nothing if dryrun - only logging
                # TODO: move dryrun logic into make_auth_req function
                if not self.dryrun:
                    logging.info("Url: {0}, Action: put, Content: {1}".format(
                        update_nanny_instances_url.format(nanny_url=self.nanny_url,
                                                          servicename=service),
                        gencfg_service_groups_decluster[service]))
                    ret = self.make_auth_req(update_nanny_instances_url.format(nanny_url=self.nanny_url,
                                                                               servicename=service),
                                             token=self.oauth_token,
                                             put=gencfg_service_groups_decluster[service],
                                             timeout=900)
                else:
                    logging.info("Dryrun fake. Url: {0}, Action: put, Content: {1}".format(
                        update_nanny_instances_url.format(nanny_url=self.nanny_url,
                                                          servicename=service),
                        gencfg_service_groups_decluster[service]))
                    # we MUST return snapshot, for dryrun purposes
                    ret = '{"runtime_attrs":{"_id": "dry_run_test_data!"}}'

                if not ret:
                    raise common.errors.TaskFailure("Nanny request failed. "
                                                    "Update instances to decluster revision failed.")

                logging.info("Step 2.1 - update gencfg instances for nanny group {0} completed. Revision:{1}".format(service,
                                                                                                                     self.mage_last_rev, ))
                snapshots_for_activate[service] = json.loads(ret)['runtime_attrs']['_id']

        except Exception as e:
            raise common.errors.TaskFailure("Cannot decluster hosts. Aborting. Error:{0}".format(e))

        logging.info("Decluster Phase 2 is finished. "
                     "Hosts going to update now. Snapshot for activate: {0}".format(json.dumps(snapshots_for_activate)))

        # return snapshot id dict for activating
        return snapshots_for_activate


class DeclusterPreparePhase3(ReclusterCtx):
    def __init__(self, oauth_token=False, mage_last_rev=False, snapshots_for_activate={}, dryrun=True):
        logging.info("DelusterPreparePhase3 prepare initialization")
        super(DeclusterPreparePhase3, self).__init__()
        self.oauth_token = oauth_token
        self.snapshots_for_activate = snapshots_for_activate
        self.mage_last_rev = mage_last_rev
        self.dryrun = dryrun

    def run(self):
        # Decluster phase 3 - activate reclustered instances

        logging.info("Decluster Phase 3 is starting.")
        logging.info("Snapshot for activate: {0}".format(self.snapshots_for_activate))

        activate_nanny_url = "{nanny_url}services/{servicename}/events/"
        try:
            for service in self.nanny_groups:
                logging.info("Try to activate service {0} with snapshot_id {1}.".format(service,
                                                                                        self.snapshots_for_activate[service]))

                post = {
                    'content': {
                        'snapshot_id': "{0}".format(self.snapshots_for_activate[service]),
                        'state': 'ACTIVE',
                        'comment': 'Sandbox auto task. Decluster. Activate {0} snapshot!'.format(
                            self.snapshots_for_activate[service]),
                        'recipe': "common"},
                    'type': 'SET_SNAPSHOT_STATE'
                }

                if not self.dryrun:
                    logging.info("Url: {0}, Action: put, Content: {1}".format(
                        activate_nanny_url.format(nanny_url=self.nanny_url,
                                                  servicename=service),
                        post))
                    ret = self.make_auth_req(activate_nanny_url.format(nanny_url=self.nanny_url,
                                                                       servicename=service),
                                             token=self.oauth_token,
                                             post=post,
                                             timeout=900)
                else:
                    logging.info("Dryrun fake. Url: {0}, Action: put, Content: {1}".format(
                        activate_nanny_url.format(nanny_url=self.nanny_url,
                                                  servicename=service),
                        post))

                    ret = True
                if not ret:
                    raise common.errors.TaskFailure("Nanny request failed. "
                                                    "Cannot SET_SNAPSHOT_STATE to ACTIVE for nanny groups.")

                logging.info("Activating for service {0} with snapshot_id {1} completed.".format(service,
                                                                                                 self.snapshots_for_activate[service]))

        except Exception as e:
            raise common.errors.TaskFailure("Cannot decluster hosts. Aborting. Error:{0}".format(e))
        logging.info("Recluster Phase 3 is finished. Hosts going to update now.")

        # Set that decluster is finished, if we cannot set it - recluster will be not started.

        if not self.dryrun:
            mage_set_last_rev_status = self.set_mage_last_revision(self.mage_url,
                                                                   self.mage_last_rev,
                                                                   'finished')
        else:
            logging.info("Dryrun fake. Set mage_set_last_status to:{0}".format('finished'))
            mage_set_last_rev_status = "dryrun"

        if not mage_set_last_rev_status:
            logging.info('Cannot set mage rev status to finished. Exiting. Details:{0}'
                         .format(mage_set_last_rev_status))
            raise common.errors.TaskFailure("Cannot set mage rev status to finished. "
                                            "Exiting. "
                                            "Details:{0}".format(mage_set_last_rev_status))

        return True
