import logging

from sandbox import sdk2
from sandbox import common
from sandbox.sandboxsdk import environments

from steps import ReclusterPreparePhase1, ReclusterPreparePhase2, ReclusterPreparePhase3


class DiskRecluster(sdk2.Task):
    """ Mail Recluster task - do recluster for disk search in RTC cloud. """

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        kill_timeout = 3600

        # custom parameters
        name = sdk2.parameters.String("Your name", default="okkk", required=True)
        secret_name = sdk2.parameters.String("Secret name", default="mage_oauth", required=True)
        dryrun = sdk2.parameters.Bool("Dryrun", default=True, required=True)
        force = sdk2.parameters.Bool("Force recluster (rewrite recluster searchmap if exists)", default=False, required=True)

        telegram_token_vault_name = sdk2.parameters.String("TelegramBot Token vault name", default="telegram_token",
                                                           required=True)
        chat_id = sdk2.parameters.Integer("Chat ID", default=-1001222981673, required=True)

    class Requirements(sdk2.Task.Requirements):
        environments = (environments.PipEnvironment('requests'),)

    def on_execute(self):
        oauth_token = sdk2.Vault.data(self.Parameters.name, self.Parameters.secret_name)
        self.telegram_token = sdk2.task.Vault.data(self.author, self.Parameters.telegram_token_vault_name)

        bot = common.telegram.TelegramBot(self.telegram_token)

        bot.send_message(self.Parameters.chat_id, "Sandbox task STARTED: {what} "
                                                  "Description: {desc} "
                                                  "Owner: {owner} "
                                                  "Force mode (rewrite recluster if exists): {force} "
                                                  "Dryrun: {dryrun}".format(what=str(self.__class__.__name__),
                                                                            desc=self.Parameters.description,
                                                                            owner=self.Parameters.owner,
                                                                            force=self.Parameters.force,
                                                                            dryrun=self.Parameters.dryrun))

        logging.info("Hello, %s!", self.Parameters.name)

        try:
            Phase1 = ReclusterPreparePhase1(oauth_token=oauth_token,
                                            rewrite_recluster=self.Parameters.force,
                                            dryrun=self.Parameters.dryrun)
            gencfg_last_rev = Phase1.run()

            Phase2 = ReclusterPreparePhase2(oauth_token=oauth_token,
                                            gencfg_last_rev=gencfg_last_rev,
                                            dryrun=self.Parameters.dryrun)

            snapshots_for_activate = Phase2.run()

            Phase3 = ReclusterPreparePhase3(oauth_token=oauth_token,
                                            gencfg_last_rev=gencfg_last_rev,
                                            snapshots_for_activate=snapshots_for_activate,
                                            dryrun=self.Parameters.dryrun)

            Phase3.run()

        except Exception as e:
            logging.info("Exception reached. Exception: {0} Detail:{1}".format(e, e.message))
            bot.send_message(self.Parameters.chat_id,
                             "Sandbox task ERROR: {what} Exception: {error}".format(what=str(self.__class__.__name__),
                                                                                    error=e))
            raise common.errors.TaskFailure("Exception reached. Exception: {0} Detail:{1}"
                                            .format(e, e.message))

        logging.info(
            "{0} completed with SUCCESS! "
            "Now another task migrates the cluster to the recluster revision.".format(str(self.__class__.__name__)))

        bot.send_message(self.Parameters.chat_id,
                         "Sandbox task finished with SUCCESS: {what}".format(what=str(self.__class__.__name__)))
