import logging
import json, time
import requests
from requests.packages.urllib3.exceptions import InsecureRequestWarning
from copy import deepcopy

requests.packages.urllib3.disable_warnings(InsecureRequestWarning)


class ReclusterCtx(object):

    def __init__(self):
        self.url_tries = 3
        self.url_sleep = 5

        self.gencfg_url = "http://api.gencfg.yandex-team.ru/"
        self.nanny_url = "https://nanny.yandex-team.ru/v2/"
        self.mage_url = "http://mage.n.yandex-team.ru/api/v3.0/"

        self.project = "disk"

        self.searchmap_prefix = "searchmap_split"

        self.revision_filename = "revision_disk.txt"

        # turn off in production? Or not - if we in 'started' state we dont make recluster in any case.
        self.rewrite_recluster = False

        self.oauth_token = False
        self.nanny_groups = ["disk_search_backend_prestable", "disk_search_backend_prod"]
        self.push_rev_nanny_groups = ["disk_search_proxy_prod",
                                      "disk_search_producer_prod",
                                      "disk_search_producer_prod_2",
                                      "disk_search_proxy_mass",
                                      "ps_producer",
                                      "mailsearch_static",
                                      "shivaka_disk",
                                      "mail_search_webtools",
                                      "disk_search_producer_yp_prod"
                                      ]

    def make_auth_req(self,
                      url,
                      token=None,
                      put=False,
                      post=False,
                      timeout=900,
                      tries=1,
                      sleep=5):

        if put and post:
            logging.info("Put and Post params defined. Error.")
            return False

        if token:
            headers = {"Authorization": "OAuth {0}".format(token),
                       "Content-Type": "application/json"}
        else:
            headers = {}

        attempts = 0
        while attempts < tries:
            if put:
                result = requests.put(url, headers=headers, json=put, timeout=timeout, verify=False)
            elif post:
                result = requests.post(url, headers=headers, json=post, timeout=timeout, verify=False)
            else:
                result = requests.get(url, headers=headers, timeout=timeout, verify=False)
            logging.info("status code: {0}".format(result.status_code))
            if result.status_code == 200 or result.status_code == 202:
                break
            else:
                logging.info("error: {0} error text:{1}".format(result.status_code, result.text))
                attempts += 1
            if attempts == tries:
                logging.info("Tries number reached")
                return False
            time.sleep(sleep)
        return result.text

    def get_gencfg_last_revision(self, gencfg_url, timeout=20):
        data = self.make_auth_req("{0}{1}/tags".format(gencfg_url, "unstable"), timeout=timeout)
        dataj = json.loads(data)['displayed_tags']
        return dataj[2]

    def get_gencfg_releases(self, nanny_service_list, timeout=20):
        g_groups = {}
        for serviceName in nanny_service_list:
            g_groups[serviceName] = set()
            data = self.make_auth_req("{0}services/{1}/".format(self.nanny_url, serviceName), token=self.oauth_token, timeout=timeout)
            service_dict = json.loads(data)
            for groupObj in service_dict['runtime_attrs']['content']['instances']['extended_gencfg_groups']['groups']:
                g_groups[serviceName].add(groupObj['release'])
        return g_groups

    def get_gencfg_releases_full(self, nanny_service_list, timeout=20):
        # returns dict with list with dicts from extended_gencfg_groups
        g_groups_full = {}
        for serviceName in nanny_service_list:
            data = self.make_auth_req("{0}services/{1}/runtime_attrs/instances/".format(self.nanny_url, serviceName), token=self.oauth_token, timeout=timeout)
            service_dict = json.loads(data)
            g_groups_full[serviceName] = service_dict

        return g_groups_full

    def check_nany_active_sn(self, curr_snapshot_id, servicename, timeout=20):
        data = self.make_auth_req("{0}services/{1}/current_state/".format(self.nanny_url, servicename),
                                  token=self.oauth_token, timeout=timeout)
        revdict = json.loads(data)
        for sn in revdict['content']['active_snapshots']:
            if sn["snapshot_id"] == curr_snapshot_id:
                # logging.info("Service {0} Rev {1} founded. State is: {2}".format(servicename,
                #                                                                 sn["snapshot_id"],
                #                                                                 sn["state"]))
                if sn["state"] == "ACTIVE" or sn["state"] == "ACTIVATING":
                    logging.info("Service {0} State for rev {1} is {2}. Test passed.".format(servicename,
                                                                                             sn["snapshot_id"],
                                                                                             sn["state"]))
                    return True
                else:
                    logging.info("Service {0} State for rev {1} is {2} (must be ACTIVE or ACTIVATING). "
                                 "Test FAILED.".format(servicename,
                                                       sn["snapshot_id"],
                                                       sn["state"]))
                    return False

        logging.info("Cant found rev {1} in service {0}. Test is FAILED".format(servicename,
                                                                                curr_snapshot_id))
        return False

    def check_nanny_active(self, nanny_service_list, timeout=20):
        for serviceName in nanny_service_list:
            data = self.make_auth_req("{0}services/{1}/runtime_attrs/".format(self.nanny_url, serviceName),
                                      token=self.oauth_token, timeout=timeout)

            dataj = json.loads(data)
            logging.info("Data {0}".format(data))
            if not self.check_nany_active_sn(dataj["_id"], serviceName):
                return False
        return True

    def get_mage_last_revision(self, mage_url, timeout=20):
        data = self.make_auth_req("{0}{1}/lastrev".format(mage_url,
                                                          self.project), timeout=timeout)

        dataj = json.loads(data)
        if dataj['status'] != 'success':
            return False
        return dataj['data']

    def set_mage_last_revision(self, mage_url, revision, status, timeout=20):
        data = self.make_auth_req("{0}{1}/lastrev?status={2}&revision={3}".format(mage_url,
                                                                                  self.project,
                                                                                  status,
                                                                                  revision), timeout=timeout)
        dataj = json.loads(data)
        if dataj['status'] != 'success':
            return False
        return dataj['message']

    @staticmethod
    def check_gencfg_nanny_releases(releases):
        ret_set = set()
        for release in releases.values():
            ret_set.update(release)
        if len(ret_set) > 1:
            return False
        else:
            return ret_set.pop().split("/")[1]

    @staticmethod
    def revisions_comparsion(left, right):
        if left.startswith("tags"):
            left = left.split("/")[1]

        if right.startswith("tags"):
            right = right.split("/")[1]

        left_tag = left.split("-")[1]
        # remove "r"
        left_rev = filter(str.isdigit, str(left.split("-")[2]))

        right_tag = right.split("-")[1]
        # remove "r"
        right_rev = filter(str.isdigit, str(right.split("-")[2]))

        if int(right_tag) > int(left_tag):
            return True

        if int(right_tag) == int(left_tag):
            if int(left_rev) < int(right_rev):
                return True
            else:
                return False

        if int(right_tag) < int(left_tag):
            return False

    @staticmethod
    def revisions_comparsion_sorted(left, right):
        if left.startswith("tags"):
            left = left.split("/")[1]
        if right.startswith("tags"):
            right = right.split("/")[1]
        left_tag = left.split("-")[1]
        left_rev = filter(str.isdigit, str(left.split("-")[2]))
        right_tag = right.split("-")[1]
        right_rev = filter(str.isdigit, str(right.split("-")[2]))

        if int(right_tag) > int(left_tag):
            return 1
        if int(right_tag) == int(left_tag):
            if int(left_rev) < int(right_rev):
                return 1
            else:
                return -1
        return -1

    def add_nanny_revision(self, gencfg_service_groups, revision):
        """
        1. Get set with uniq releases from nanny group
        2. Get set with uniq groups from nanny group
        3. Get most bigger release
        4. Copy config (cpu, limits, etc) from bigger release, from group and change revision
        5. Add it to  groups_toadd dict.
        6. Add all groups from groups_toadd to gencfg_service_groups, return it.
        """
        releases_dict_set = {}
        groups_dict_set = {}
        groups_toadd = {}

        copied_gencfg_service_groups = deepcopy(gencfg_service_groups)

        for service, service_data in gencfg_service_groups.iteritems():
            releases_dict_set[service] = set()
            groups_dict_set[service] = set()

            groups_toadd[service] = []
            for group in service_data['content']['extended_gencfg_groups']['groups']:
                releases_dict_set[service].add(group['release'])
                groups_dict_set[service].add(group['name'])
            actual_service_release = sorted(list(releases_dict_set[service]), cmp=self.revisions_comparsion_sorted)[0]

            if "tags/{0}".format(revision) in releases_dict_set[service]:
                return False

            for group_uniq in groups_dict_set[service]:
                for group in service_data['content']['extended_gencfg_groups']['groups']:
                    if group_uniq == group['name'] and group['release'] == actual_service_release:
                        group_copy = deepcopy(group)
                        group_copy['release'] = "tags/{0}".format(revision)
                        groups_toadd[service].append(group_copy)

            copied_gencfg_service_groups[service]['content']['extended_gencfg_groups']['groups'] += groups_toadd[service]
        return copied_gencfg_service_groups

    def remove_smallest_nanny_revision(self,
                                       gencfg_service_groups,
                                       revision):
        """
        remove all releases except specified revision
        """

        releases_dict_set = {}
        releases_min_set = set()
        copied_gencfg_service_groups = deepcopy(gencfg_service_groups)

        for service, service_data in gencfg_service_groups.iteritems():
            if not releases_dict_set.get(service, False):
                releases_dict_set[service] = set()
            for group in service_data['content']['extended_gencfg_groups']['groups']:
                releases_dict_set[service].add(group['release'])

            # so we have here smallest release (use reverse) for nanny service. Also we truncate "tags/" prefix here.
            service_release = sorted(list(releases_dict_set[service]),
                                     cmp=self.revisions_comparsion_sorted,
                                     reverse=True)[0]

            if service_release.split("/")[1] == revision:
                logging.info("ERROR! Decluster task cannot remove last revision from services, last revision is equal"
                             "to smallest rev in groups. Service:{0}, rev in group: {1}, last revison to delete: {2}".format(service,
                                                                                                                             service_release,
                                                                                                                             revision))
                return False

            # strip "/tags"
            releases_min_set.add(service_release.split("/")[1])

        if len(releases_min_set) > 1:
            logging.info("ERROR! We have two different min revs for nanny groups. Exiting. Details:{0}".format(str(releases_min_set)))
            return False

        min_rev = releases_min_set.pop()
        # so we have checked min rev here - we need pop it from gencfg_service_groups dict and return it

        for service, service_data in gencfg_service_groups.iteritems():
            for group in service_data['content']['extended_gencfg_groups']['groups']:
                if group['release'] == "tags/{0}".format(min_rev):
                    copied_gencfg_service_groups[service]['content']['extended_gencfg_groups']['groups'].remove(group)
        return copied_gencfg_service_groups

    def yasm_check(self, ctype):
        kamaji_5xx_rate = 100
        index_copy_rate = 100
        index_progress_perc = 99

        host = "ASEARCH"
        period = 300
        et = time.time() - period * 5
        st = et - period * 5
        signals = ["itype=bacchus;ctype={ctype};prj=disk-main:index-indexer-codes-5xx_rps_ammv".format(ctype=ctype),
                   "itype=bacchus;ctype={ctype};prj=disk-main:index-index-copy-current-rate-mb_ammv".format(ctype=ctype),
                   "itype=bacchus;ctype={ctype};prj=disk-main:perc(index-index-copy-progress_ammt, or(index-instance-alive_ammt,index-instance-alive_ammx))".format(ctype=ctype)]

        maxdict = {}

        for timestamp, values in self.yasmapi(host, period, st, et, signals):
            logging.info("Curr values:{0}".format(values))
            for key, data in values.iteritems():
                if not maxdict.get(key, False):
                    maxdict[key] = data
                if maxdict[key] < int(data) \
                        and (key == signals[0] or key == signals[1]):
                    maxdict[key] = int(data)
                if (maxdict[key] > data and key == signals[2]):
                    maxdict[key] = data

        if maxdict[signals[0]] > kamaji_5xx_rate:
            logging.info("Kamaji 5xx rate is {0}. Its bigger than: {1} Decluster cannot continue.".format(
                maxdict[signals[0]],
                kamaji_5xx_rate))
            return False
        if maxdict[signals[1]] > index_copy_rate:
            logging.info("Index copy rate rate is {0}. Its bigger than: {1} Decluster cannot continue.".format(
                maxdict[signals[1]],
                index_copy_rate))
            return False
        if maxdict[signals[2]] < index_progress_perc:
            logging.info("Index copy perc rate is {0}. Its smaller than: {1} Decluster cannot continue.".format(
                maxdict[signals[2]],
                index_progress_perc))
            return False

        for signal, data in maxdict.iteritems():
            logging.info("Key is: {0} Rate for key: {1}".format(signal, data))

        logging.info("Yasm {ctype} checks done. Decluster continues.".format(ctype=ctype))
        return True
