import logging
import os
import shutil
import zipfile

from sandbox import sdk2
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
from sandbox.common.errors import TaskFailure
from sandbox.sandboxsdk.environments import PipEnvironment


COMMON_TEAMCITY = 'https://teamcity.yandex-team.ru'
TAXI_TEAMCITY = 'https://teamcity.taxi.yandex-team.ru'

REPORT_FILENAME = 'allure-report.zip'


class EatsCreateTestpalmRun(sdk2.Task):
    class Parameters(sdk2.Parameters):
        testpalm_project_id = sdk2.parameters.String(
            'Testpalm project id', required=True,
        )
        testpalm_suite_id = sdk2.parameters.String(
            'Testpalm suite id', required=True,
        )
        testpalm_run_id = sdk2.parameters.String('Testpalm run id')
        testrun_title = sdk2.parameters.String(
            'Testrun title', default='CI Automated', required=True,
        )
        test_build_id = sdk2.parameters.Integer('Test build id', required=True)
        with sdk2.parameters.RadioGroup('Teamcity') as teamcity:
            teamcity.values[COMMON_TEAMCITY] = teamcity.Value(
                value=COMMON_TEAMCITY,
            )
            teamcity.values[TAXI_TEAMCITY] = teamcity.Value(
                value=TAXI_TEAMCITY, default=True,
            )

    class Requirements(sdk2.Task.Requirements):
        disk_space = 10 * 1024
        client_tags = (
            ctc.Tag.MULTISLOT | ctc.Tag.GENERIC
        ) & ctc.Tag.Group.LINUX
        cores = 1
        ram = 2048
        dns = ctm.DnsType.DNS64
        environments = [
            PipEnvironment('testpalm-api-client', version='4.0.2'),
            PipEnvironment('teamcity-client', version='5.0.0'),
        ]

        class Caches(sdk2.Task.Requirements.Caches):
            pass

    def report_archive_path(self):
        return str(self.path(REPORT_FILENAME))

    @property
    def teamcity_client(self):
        import teamcity_client.client
        vault_key = (
            'eats-teamcity'
            if self.Parameters.teamcity == COMMON_TEAMCITY
            else 'eats-taxi-teamcity'
        )
        return teamcity_client.client.TeamcityClient(
            server_url=self.Parameters.teamcity,
            auth=sdk2.Vault.data(vault_key),
        )

    @property
    def testpalm_client(self):
        from testpalm_api_client.client import TestPalmClient

        return TestPalmClient(oauth_token=sdk2.Vault.data('eats-testpalm'))

    def download_allure_report(self):
        build = self.teamcity_client.Build(id=self.Parameters.test_build_id)

        logging.info('Looking for allure report in {}'.format(build.web_url))
        for artifact in build.artifacts():
            if artifact.name == REPORT_FILENAME and artifact.is_file:
                artifact_fileobj = artifact.download()
                with open(self.report_archive_path(), 'wb') as f:
                    shutil.copyfileobj(artifact_fileobj, f)
                break

        if not os.path.exists(self.report_archive_path()):
            raise TaskFailure(
                'Failed to find allure report in {}'.format(build.web_url),
            )

        logging.info('Finished downloading report')

    def parse_allure_report(self):
        from sandbox.projects.eats.EatsCreateTestpalmRun.allure.allure_parser import (
            AllureParser,
        )

        return AllureParser(
            str(self.path('allure-report/data/test-cases')),
        ).prepare_data()

    def on_execute(self):
        import sandbox.projects.eats.EatsCreateTestpalmRun.testpalm as testpalm

        project_id = self.Parameters.testpalm_project_id
        testsuite_id = self.Parameters.testpalm_suite_id
        testrun_title = self.Parameters.testrun_title
        testpalm_run_id = self.Parameters.testpalm_run_id

        logging.info('Downloading allure report')
        self.download_allure_report()

        logging.info('Decompressing archive')
        with zipfile.ZipFile(self.report_archive_path(), 'r') as archive:
            archive.extractall(str(self.path()))

        logging.info('Parsing allure report')
        allure_testcases = self.parse_allure_report()
        logging.info('Test results are: %s' % allure_testcases)

        testrun = testpalm.TestpalmTestrun.init_by_api(
            testpalm_client=self.testpalm_client,
            project_id=project_id,
            testrun_id=testpalm_run_id,
            testsuite_id=testsuite_id,
            testrun_title=testrun_title,
            assignee='robot-eda-autotest',
            created_by='robot-eda-autotest',
        )
        testrun.update_from_allure_cases(allure_testcases)
