import contextlib
import os
import pathlib2
import shutil
import tarfile

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment, SandboxGitLfsEnvironment, PipEnvironment
from sandbox.projects.common.teamcity import TeamcityArtifacts, TeamcityArtifactsContext
from sandbox.sdk2.helpers import ProcessLog, subprocess
from sandbox.sdk2.vcs.git import Git
from sandbox.sdk2 import ssh


class EatsRunAndroidTests(sdk2.Task):
    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Scm') as scm:
            branch = sdk2.parameters.String('Branch')
            commit = sdk2.parameters.String('Commit', required=True)
        sdk = sdk2.parameters.Resource('Android SDK', required=True, default=2892761295)
        emulator_sdk = sdk2.parameters.Resource("Emulator sdk", required=True, default=2892761295)
        emulator_config = sdk2.parameters.Resource("Emulator config", required=True, default=2419899638)
        adbserver = sdk2.parameters.Resource('adbserver', required=True, default=2356349620)
        gradle_target = sdk2.parameters.String('Gradle target', required=True)
        artifacts_to_publish = sdk2.parameters.String('Artifacts to publish')
        tokens = sdk2.parameters.YavSecret("YAV secret", default="sec-01d2ygse80sf08pcwrhsd0pqzs")
        debug = sdk2.parameters.String('Debug', default="")
        _container = sdk2.parameters.Container("Container", default=2437845477, required=True)

    class Requirements(sdk2.Task.Requirements):
        disk_space = 30 * 1024
        client_tags = (ctc.Tag.MULTISLOT | ctc.Tag.GENERIC) & ctc.Tag.Group.LINUX & ctc.Tag.SSD
        cores = 8
        ram = 31 * 1024
        dns = ctm.DnsType.DNS64
        environments = [
            SandboxGitLfsEnvironment('2.5.2'),
            SandboxJavaJdkEnvironment('11.0.8', platform='linux'),
            PipEnvironment('yandex-avd-tools==1.5.0', custom_parameters=['--no-deps']),
            PipEnvironment('decorator>=4.4.2,<5'),
            PipEnvironment('killall>=1.1.0,<2'),
            PipEnvironment('py>=1.5.2,<2'),
            PipEnvironment('pyjavaproperties>=0.7,<1'),
            PipEnvironment('retry>=0.9.2,<1'),
            PipEnvironment('six<2'),
        ]

        class Caches(sdk2.Task.Requirements.Caches):
            pass

    def checkout_path(self, *args):
        return str(self.path('checkout-dir', *args))

    def sdk_path(self, *args):
        return str(self.path('sdk', *args))

    def checkout_code(self):
        with ssh.Key(key_name="EATS_KEY"):
            vcs_root = Git(url='ssh://git@bb.yandex-team.ru/eda/android_client.git', filter_branches=False)
            vcs_root.clone(self.checkout_path(), self.Parameters.branch, commit=self.Parameters.commit)

    def get_sdk(self):
        sdk_archive_path = str(sdk2.ResourceData(self.Parameters.sdk).path)
        with tarfile.open(sdk_archive_path, 'r:gz') as sdk_archive:
            sdk_archive.extractall(path=self.sdk_path())
        self.accept_licenses()

    def accept_licenses(self):
        with ProcessLog(self, logger='accept_licenses') as pl:
            yes = subprocess.Popen(['yes'], stdout=subprocess.PIPE)
            subprocess.check_call(
                [self.sdk_path('cmdline-tools', 'latest', 'bin', 'sdkmanager'), '--licenses'],
                stdout=pl.stdout, stderr=pl.stdout, stdin=yes.stdout)
            yes.kill()

    def run_gradle(self):
        env = os.environ.copy()
        env['ANDROID_SDK_ROOT'] = self.sdk_path()
        env['YA_ARTIFACTORY_TOKEN'] = self.Parameters.tokens.data()["ya_artifactory_token"]
        gradle_targets = self.Parameters.gradle_target.split()
        with TeamcityArtifactsContext(pathlib2.Path(self.checkout_path())) as artifacts_context:
            return subprocess.call(
                ['./gradlew'] + gradle_targets,
                cwd=self.checkout_path(), env=env, stdout=artifacts_context.output, stderr=artifacts_context.output)

    @contextlib.contextmanager
    def run_adbserver(self):
        self.debug("before_run_adbserver")
        adbserver_path = str(sdk2.ResourceData(self.Parameters.adbserver).path)
        env = os.environ.copy()
        env['ANDROID_SDK_ROOT'] = self.emulator_sdk_path()
        env['PATH'] = self.emulator_sdk_path('platform-tools') + ":" + env['PATH']
        with ProcessLog(self, logger='adbserver') as pl:
            adbserver = subprocess.Popen(['java', '-jar', adbserver_path], stdout=pl.stdout, stderr=pl.stdout, env=env)
            try:
                yield
            finally:
                adbserver.kill()

    @contextlib.contextmanager
    def run_logcat(self):
        from avd_tools import paths
        with ProcessLog(self, logger='logcat') as pl:
            logcat_process = subprocess.Popen([paths.adb_path(), 'logcat'], stdout=pl.stdout)
            try:
                yield
            finally:
                logcat_process.kill()

    def debug(self, stage):
        if self.Parameters.debug == stage or self.Parameters.debug == "all":
            self.suspend()

    def emulator_sdk_path(self, *args):
        return str(self.path('emulator-sdk', *args))

    def marathon_report_path(self, *args):
        return str(self.checkout_path('build', 'reports', 'marathon', *args))

    def get_emulator_sdk(self):
        sdk_archive_path = str(sdk2.ResourceData(self.Parameters.emulator_sdk).path)
        with tarfile.open(sdk_archive_path, 'r:gz') as sdk_archive:
            sdk_archive.extractall(path=self.emulator_sdk_path())

    def artifacts_path(self, *args):
        return str(self.path('teamcity_artifacts', *args))

    def artifacts_exist(self):
        for root, dirs, files in os.walk(self.artifacts_path()):
            if files:
                return True
        return False

    def publish_artifacts(self, files_to_publish):
        os.mkdir(self.artifacts_path())
        for filename in files_to_publish:
            abs_path = filename if os.path.isabs(filename) else os.path.abspath(self.checkout_path(filename))
            if not os.path.exists(abs_path):
                continue
            if os.path.isdir(abs_path):
                shutil.copytree(abs_path, self.artifacts_path(os.path.basename(abs_path)))
            else:
                shutil.copy(abs_path, self.artifacts_path())
        if self.artifacts_exist():
            artifacts_resource = TeamcityArtifacts(self, 'Teamcity artifacts', self.artifacts_path())
            sdk2.ResourceData(artifacts_resource).ready()

    def generate_allure_report(self, results_dir, report_dir):
        with ProcessLog(self, logger='generate_report') as pl:
            subprocess.call(["allure", 'generate', results_dir, "-o", report_dir],
                            stdout=pl.stdout, stderr=pl.stdout)
            shutil.make_archive(report_dir, 'zip', os.path.dirname(report_dir), os.path.basename(report_dir))

    def run_marathon(self):
        env = os.environ.copy()
        env['ANDROID_SDK_ROOT'] = self.emulator_sdk_path()
        with ProcessLog(self, logger='run_marathon') as pl:
            return subprocess.call(
                ["marathon", "--analyticsTracking", "false", "--bugsnag", "false",
                 "--marathonfile", "app/src/androidTest/Marathon"],
                cwd=self.checkout_path(), env=env, stdout=pl.stdout, stderr=pl.stdout)

    def on_execute(self):
        self.checkout_code()
        self.get_sdk()
        self.get_emulator_sdk()

        import avd_tools
        avd_tools.utils.kill_stale_processes()
        avd_tools.utils.remove_android_home_dir()
        avd_tools.paths.set_sdk_path(self.emulator_sdk_path())

        self.debug("before_launch_avd")
        emulator_config = str(sdk2.ResourceData(self.Parameters.emulator_config).path)
        allure_report_path = str(self.path("allure-report"))
        marathon_allure_report_path = str(self.path("marathon-allure-report"))
        try:
            gradle_exit_code = self.run_gradle()
            if gradle_exit_code:
                raise TaskFailure('Gradle exited with {}'.format(gradle_exit_code))

            with self.run_adbserver(), avd_tools.RunManyAvdWithRetry(emulator_config, tries=1, count=4), self.run_logcat():
                marathon_exit_code = self.run_marathon()

            marathon_results_path = str(self.marathon_report_path("allure-results"))
            self.generate_allure_report(marathon_results_path, marathon_allure_report_path)

            allure_results_path = str(self.marathon_report_path("device-files", "allure-results"))
            self.generate_allure_report(allure_results_path, allure_report_path)

            if marathon_exit_code:
                raise TaskFailure('Marathon exited with {}'.format(marathon_exit_code))
        finally:
            self.publish_artifacts(self.Parameters.artifacts_to_publish.split() + [
                str(self.log_path("adbserver.out.log")),
                str(self.log_path("logcat.out.log")),
                allure_report_path + '.zip',
                self.marathon_report_path(),
                marathon_allure_report_path + '.zip'
            ])
        self.debug("finish")
