# coding=utf-8

import json
import logging

from collections import defaultdict
from typing import Callable, Dict, Iterable, List, Optional, Union

import pandas as pd

from startrek_client import Startrek
from yt.wrapper import YtClient

from . import utils


TABLE_NAME = 'Statistic2.0'
TABLE_PATH = '//home/eda/statistics/'

# it's attributes of tickets needed for statistic
# if the attribute is dict, it means that it has nested values
task_attributes: List[Union[str, Dict[str, Union[str, List[str]]]]] = [
    {'abcService': ['name']},
    {'assignee': ['display', 'login']},
    {'components': ['name']},
    'createdAt',
    {'epic': ['key']},
    'goals',
    'iterationCount',
    {'project': ['key']},
    'key',
    {'project': ['key']},
    {'resolution': ['id']},
    'resolvedAt',
    'spent',
    'spentSp',
    {'sprint': ['id', 'name', 'startDate', 'endDate']},
    'storyPoints',
    'summary',
    'tags',
    {'type': ['key']},
    'qaSpent',
]
# In the final date frame, the columns will be formed in the
# following way. For example:
# {'sprint': ['id', 'name']} -> columns 'sprint_id' and 'sprint_name'
# {'abcService': ['name']} -> column 'abcService_name'
# 'spent' -> column 'spent'


def to_float(value=0):
    value = round(float(value), 2)
    # it is filter for task with 0 Sp
    if value > 1e6:
        value = 0.0
    return value


MAPING_TABLE_TO_COLUMNS = {
    # Name Table: {Table schema}
    TABLE_NAME: {
        # schema is:
        # name in pandas.DataFrame in sandbox ->
        # Tuple of Name in bd and type casting to bd right type
        'abcService_name': ('AbcService', str),
        'components_name': ('Components', str),
        'epic_key': ('Epic', str),
        'goals': ('Goals', str),
        'iterationCount': ('IterationCount', int),
        'key': ('Key', str),
        'project_key': ('Project', str),
        'resolution_id': ('ResolutionId', int),
        'assignee_display': ('AssigneeName', str),
        'assignee_login': ('AssigneeLogin', str),
        'sprint_id': ('SprintId', int),
        'sprint_name': ('SprintName', str),
        'sprint_startDate': ('SprintStart', int),
        'sprint_endDate': ('SprintEnd', int),
        'storyPoints': ('storyPoints', to_float),
        'spent': ('Spent', to_float),
        'spentSp': ('spentSp', to_float),
        'tags': ('Tags', str),
        'SumOverallTime': ('SumOverallTime', to_float),
        'TimeForOneSp': ('TimeForOneSp', to_float),
        'OneDayOnOneTaskTypeInSp': ('OneDayOnOneTaskTypeInSp', to_float),
        'WeekOnOneTaskTypeInSp': ('WeekOnOneTaskTypeInSp', to_float),
        'TimeForOneTask': ('TimeForOneTask', to_float),
        'SpInHour': ('SpInHour', to_float),
        'type': ('TypeSpeed', int),
        'type_15': ('TypeSpeed15', int),
        'Command': ('Command', str),
        'type_key': ('Type', str),
        'SpentQA': ('SpentQA', float),
        'summary': ('Summary', str),
        'createdAt': ('CreationTime', int),
        'resolvedAt': ('ResolutionTime', int),
    },
}

VALUE_COLUMNS = [
    x[0] for x in MAPING_TABLE_TO_COLUMNS[TABLE_NAME].values() if x[0] != 'Key'
]


class StartrekQuery2DataFrame:
    """
    Class for using api startrek client and transform attributes of tasks into
    pandas.DataFrame
    """

    def __init__(self, startrek_api: Startrek) -> None:
        """
        Init function
        Args:
            startrek_api: simple startrek api
        """
        self._startrek_api = startrek_api

    def _get_tickets_iterator(self, query: str) -> Iterable:
        """
        Function for getting tickets from startrek client
        Args:
            query:
        Returns:
            generator of tickets
        """
        return (ticket for ticket in self._startrek_api.issues.find(query))

    def serialize(
            self,
            obj: object,
            needed_attributes: List[
                Union[str, Dict[str, Union[str, List[str]]]]
            ],
            prefix: str = '',
    ) -> Dict[str, Union[str, int, float, List[str]]]:
        """
        Function for create dict from object
        Args:
            obj: input object
            needed_attributes: needed attributes of object
            prefix: nesting level
        Returns:
            object representation as python dict
        """
        obj_attrs = {}
        for attr in needed_attributes:
            # get attribute with one value (attribute is simple object)
            if isinstance(attr, str):
                attr_key = prefix + attr
                if isinstance(obj, (tuple, list)):
                    obj_attrs[attr_key] = [
                        utils.get_attr(one_obj, attr) for one_obj in obj
                    ]
                else:
                    obj_attrs[attr_key] = utils.get_attr(obj, attr)
            elif isinstance(attr, dict):
                # get nested attributes
                for outer_attr in attr:
                    obj_attrs.update(
                        self.serialize(
                            utils.get_attr(obj, outer_attr),  # inner object
                            attr[outer_attr],  # get nested attributes
                            outer_attr + '_'  # separating the nested
                            # attribute name with _
                        ),
                    )
        return obj_attrs

    def count_tickets(self, query: str) -> int:
        """
        Count ticket by query
        Args:
            query: query for startrek
        Returns:
            count tickets
        """
        return len(self._startrek_api.issues.find(query))

    def create_df(self, query: str) -> pd.DataFrame:
        """
        Function for creating pandas.DataFrame from many startrek tickets
        Args:
            query: query to startrek
        Returns:
            tickets as dataframe
        """
        dataframe = defaultdict(list)
        for ticket in self._get_tickets_iterator(query):
            ticket_attrs = self.serialize(ticket, task_attributes)
            for key in ticket_attrs:
                dataframe[key].append(ticket_attrs[key])
        logging.info('Dataframe has columns: {}', list(dataframe.keys()))
        return pd.DataFrame(dataframe)


class YtWriter:
    def __init__(self, yt_api: YtClient) -> None:
        self._yt_api = yt_api
        self._overall_tables = []

    def add_statistic(self, statistic: pd.DataFrame) -> None:
        """
        Function for add statistic in
        Args:
            statistic: pd.DataFrame
        """
        if statistic.shape[0] > 0:
            self._overall_tables.append(statistic)

    def write(self, append: Optional[bool] = True):
        statistic = pd.concat(self._overall_tables)
        self._write_in_bd(statistic, TABLE_NAME, append)
        self._overall_tables = []

    @property
    def count_rows(self):
        return sum(df.shape[0] for df in self._overall_tables)

    def _write_in_bd(self, dataframe, table_name, append):
        if table_name not in MAPING_TABLE_TO_COLUMNS:
            logging.debug('Can\'t write into table {}.'.format(table_name))
            return
        schema = MAPING_TABLE_TO_COLUMNS[table_name]
        data_for_table = []
        path = TABLE_PATH + table_name
        if append:
            path = '<append=%true>' + path
        for i in range(dataframe.shape[0]):
            data_for_table.append(
                self._create_row(
                    dataframe.iloc[i, :], schema, dataframe.columns,
                ),
            )
        logging.info('Writing data to `{}`'.format(path))
        logging.info(
            'Examples of data to be inserted: {}'.format(
                json.dumps(data_for_table[:3]),
            ),
        )
        self._yt_api.write_table(path, data_for_table, raw=False)

    @staticmethod
    def _create_row(
            row: pd.Series, schema: Dict[str, Callable], columns: List[str],
    ) -> dict:
        """
        Method for transform row dataframe into row for table of bd
        Args:
            row: row in dataframe with statistic
            schema: dict - name column in dataframe value - (name column in
                table, needed type)
            columns: columns of pandas.DataFrame
        Returns:
            row in right form
        """
        data = {}
        for col in columns:
            value = row[col]
            if col in schema:
                name_in_table, type_for_table = schema[col]
                value = (
                    type_for_table()
                    if pd.isnull(value)
                    else type_for_table(value)
                )
                data[name_in_table] = value
        return data
