# coding=utf-8
import logging

from typing import Any, Dict, List, Optional

import pandas as pd

from . import clients, models, tables, utils, unduplicator


def create_list_statistic(kwargs: dict):
    return [(tables.TableWithSpAndHourStatisticSprints, kwargs)]


def compute_statistic(data: pd.DataFrame, statistics: List[Any]):
    """
    Function for compute and write tables
    Args:
        data: input data
        statistics: Sequence of tables and kwargs to it
    Returns:
        last table or None
    """
    table = None
    if statistics:
        for i, (handler_initializer, kwargs) in enumerate(statistics):
            try:
                handler = handler_initializer(data, True)
                table = handler.compute_statistic(**kwargs)
                shape = table.shape
                logging.info(
                    f'Shape of table is {shape[0]} rows {shape[1]} columns',
                )
            except Exception as err:
                logging.info(f'Caught Exception: {err}')
                raise
        logging.info('The process is over')
    return table


def main(
        client_yt,
        client_startrek,
        client_yql,
        analyzed_team: Optional[Dict[str, str]] = None,
):
    yt_writer = clients.YtWriter(client_yt)
    configurator = models.Configurator(analyzed_team)
    startrek = clients.StartrekQuery2DataFrame(client_startrek)
    for team_name, query, filter, params in configurator.create_run():
        logging.info(f'Query for startrek is: {query}')
        acceptable_tags = params.pop('acceptable tags')
        invalid_tags = params.pop('invalid tags')
        data = utils.prepare_dataframe(
            startrek.create_df(query),
            ['resolution_id'],
            ['storyPoints', 'spentSp'],
            acceptable_tags,
            invalid_tags,
        )
        if data.shape[0] == 0:
            logging.info(f'Found no tasks for command {team_name}')
            continue
        logging.info(f'Found {data.shape[0]} tasks for group {team_name}')
        filtered_data = filter(data)
        if filtered_data.shape[0] == 0:
            logging.info(
                f'Found no tasks for command {team_name} after filtering',
            )
            continue
        logging.info(f'After filtering - {filtered_data.shape[0]} tasks')
        list_tables = create_list_statistic(params)
        result = compute_statistic(filtered_data, list_tables)
        if result is not None:
            result.loc[:, 'Command'] = team_name
            result['SpentQA'] = filtered_data.qaSpent.apply(utils.string2time)
            yt_writer.add_statistic(result)
    logging.info(f'Overall task count is {yt_writer.count_rows}')

    if yt_writer.count_rows > 0:
        yt_writer.write(append=True)

    unduplicator.remove_duplicates(
        client_yql,
        clients.TABLE_PATH + clients.TABLE_NAME,
        clients.VALUE_COLUMNS,
    )
