# coding=utf-8

from datetime import datetime, timedelta
from typing import Any, Dict, List, Optional, Sequence, Tuple, Union

import pandas as pd

from . import config, utils

COLUMNS_WITH_MANY_VALUES = [
    'abcService_name', 'components_name', 'goals', 'tags'
]


class Configurator:
    """
    Class for creating statistic computing by input parameters and config
        and check config
    """
    def __init__(
        self, analyzed_teams: Union[None, Dict[str, Tuple[int, int, int]]]
    ):
        self._analyzed_teams = analyzed_teams
        self._config = config.teams
        if analyzed_teams is not None:
            if list(analyzed_teams.keys())[0] not in self._config:
                raise ValueError(
                    '''The name of the team is incorrect.'''
                    f'''Set one of {list(self._config.keys())}'''
                )

    def create_query_with_resolve(self, team_name, start_day, end_day):
        """
        Function for creating query to startrek
        Args:
            team_name (str): name team in config
            start_day (datetime.datetime): start date for analysis
            end_day (datetime.datetime): last date for analysis
        Returns:
            query for startrek
        """
        query = self._config[team_name]['query']
        query += ' Resolved: {}-{}-{}..{}-{}-{}'.format(
            start_day.year, start_day.month, start_day.day,
            end_day.year, end_day.month, end_day.day
        )
        return query

    def create_date(self):
        """
        Function for create start and end day for analysis
        Returns:
            tuple of two datetime (start, end)
        """
        scheduler_starts = [0, 4]  # index of Monday and Friday
        today = datetime.today()
        weekday = today.weekday()
        if weekday not in scheduler_starts and self._analyzed_teams is None:
            raise ValueError(
                f'Strange day to starting script. '
                f'{today.year}-{today.month}-{today.day}'
            )
        if self._analyzed_teams is not None:
            start = datetime(*list(self._analyzed_teams.values())[0])
            if weekday >= 4:  # days after Friday
                end = today + timedelta(days=6 - weekday)
            else:
                end = today + timedelta(days=3 - weekday)
        else:
            # starting with scheduler
            if weekday == 4:  # is Friday
                end = today - timedelta(days=1)  # Thursday
                start = today - timedelta(days=4)  # Monday
            elif weekday == 0:  # is Monday
                end = today - timedelta(days=1)  # Sunday
                start = today - timedelta(days=3)  # Friday
        return start, end

    def create_run(self):
        """
        Function for iterating over config
        Returns:
            (team name, query, filter function, parameters)
        """
        start, end = self.create_date()
        if self._analyzed_teams is not None:
            team_names = self._analyzed_teams.keys()
        else:
            team_names = self._config.keys()
        for team_name in team_names:
            yield (
                team_name,
                self.create_query_with_resolve(team_name, start, end),
                Filter(self._config[team_name].get('filters', None)),
                self.group_parameters(team_name)
            )

    def group_parameters(self, team_name):
        params = {
            key: self._config[team_name][key]
            for key in self._config[team_name]
            if key not in ['query', 'filters']
        }
        for key in ['acceptable tags', 'invalid tags']:
            if key not in params:
                params[key] = None
        return params


class Filter:
    def __init__(self, filters: Optional[Dict[str, List[Any]]]):
        # default filters
        self._filters = [
            {'resolution_id': [1]}
        ]
        if filters:
            for column in filters:
                self._filters.append(
                    {column: filters[column]}
                )

    def __call__(self, data: pd.DataFrame) -> Union[None, pd.DataFrame]:
        data = self._filter_not_in(data, {'storyPoints': [0]})
        for column_and_value in self._filters:
            column = list(column_and_value.keys())[0]
            if column in COLUMNS_WITH_MANY_VALUES:
                data = self._filter_by_column_values(data, column_and_value)
            else:
                data = self._filter_by_column_value(data, column_and_value)
        return data

    @staticmethod
    def _filter_by_column_value(
        data: pd.DataFrame,
        column_and_value: Optional[Dict[str, List[Any]]] = None,
        inverse: Optional[bool] = False
    ) -> pd.DataFrame:
        """
        Method for filter data by some column. Return rows where value from column is
            in accepted values
        Args:
            data: input data
            column_and_value: dict with keys: columns values: it's accepted values
        Returns: filtered data

        """
        if column_and_value:
            index = True
            for key in column_and_value:
                index &= data[key].isin(column_and_value[key])
            if inverse:
                return data[~index]
            else:
                return data[index]
        else:
            return data

    @staticmethod
    def _filter_not_in(
        data: pd.DataFrame,
        column_and_value: Optional[Dict[str, List[Any]]] = None,
        inverse: Optional[bool] = False
    ) -> pd.DataFrame:
        """
        Method for filter data by some column. Return rows where value from column is
            not in accepted values
        Args:
            data: input data
            column_and_value: dict with keys: columns values: it's accepted values
        Returns: filtered data

        """
        if column_and_value:
            index = True
            for key in column_and_value:
                index &= ~data[key].isin(column_and_value[key])
            if inverse:
                return data[~index]
            else:
                return data[index]
        else:
            return data

    @staticmethod
    def _filter_by_column_values(
        data: pd.DataFrame,
        column_and_values: Optional[Sequence[Any]] = None
    ) -> pd.DataFrame:
        """
        Method for filter data by some column. Return rows where data from column contain
            accepted sequence of values
        Args:
            data: input data
            column_and_values: dict with keys: columns values: it's accepted
                sequence of values
        Returns: filtered data
        """
        if column_and_values:
            index = True
            for key in column_and_values:
                index &= data[key].apply(
                    lambda x: all([
                        el in x for el in column_and_values[key]
                    ])
                )
            return data[index]
        else:
            return data


class BaseTableHandler:
    """
    Base class for computing statistic on dataframe
    """
    def __init__(
        self,
        data: pd.DataFrame,
        transform_hours_to_float: Optional[bool] = True,
    ) -> None:
        """
        Init function
        Args:
            data: input dataframe
            transform_hours_to_float: is transform column
                'spent' in hours
        """
        # copy dataframe (because pandas.DataFrame is mutable object)
        copy_data = data.copy()
        if 'spent' in copy_data.columns and transform_hours_to_float:
            copy_data['spent'] = copy_data.spent.apply(utils.string2time)
        self.data = copy_data

    def compute_statistic(self, **kwargs) -> pd.DataFrame:
        """
        Function for computing needed statistic
        Args:
            kwargs (dict): additional parameters
        Returns:
            computed statistic in any type
        """
        raise NotImplementedError

    def count_tasks(self) -> int:
        return self.data.shape[0]
