# coding=utf-8

from typing import Optional

import pandas as pd

from . import utils
from . import models


DAY_ON_1_TASK = 'OneDayOnOneTaskTypeInSp'
COUNT_TASKS = 'CountTask'
SUM_SP = 'SumSp'
SP_IN_HOUR = 'SpInHour'
SP = 'storyPoints'
SUM_HOURS = 'SumHours'
SUM_TIME = 'SumOverallTime'
TASK_IN_SP = 'TaskInSp'
TIME_ON_1_SP = 'TimeForOneSp'
TIME_ON_1_TASK = 'TimeForOneTask'
WEEK_ON_TASK = 'WeekOnOneTaskTypeInSp'

COLUMNS_FROM_DF = [
    'abcService_name',
    'assignee_display',
    'assignee_login',
    'components_name',
    'epic_key',
    'goals',
    'iterationCount',
    'key',
    'project_key',
    'resolution_id',
    'spent',
    'spentSp',
    'sprint_id',
    'sprint_name',
    'storyPoints',
    'tags',
    'type_key',
    'sprint_startDate',
    'sprint_endDate',
    'summary',
    'createdAt',
    'resolvedAt',
]
COLUMNS_FROM_FAST = ['type', 'type_15']
COLUMNS_OVERALL_STAT = [
    'SumOverallTime',
    'TimeForOneSp',
    'OneDayOnOneTaskTypeInSp',
    'WeekOnOneTaskTypeInSp',
    'TimeForOneTask',
    'SpInHour',
]


class TableWithSpAndHourStatistic(models.BaseTableHandler):
    def compute_statistic(
            self, key_to_spent_time: Optional[str] = 'spent', **kwargs,
    ):
        data = self.data
        sum_spent_sp, sum_spent_hour = (
            data[SP].sum(),
            data[key_to_spent_time].sum(),
        )
        table = data.groupby(by=SP, as_index=False).agg(
            {key_to_spent_time: ['count', 'sum']},
        )
        table.columns = [TASK_IN_SP, COUNT_TASKS, SUM_TIME]
        try:
            table[SP_IN_HOUR] = (
                table[COUNT_TASKS] * table[TASK_IN_SP] / table[SUM_TIME]
            )
        except ZeroDivisionError:
            table[SP_IN_HOUR] = 0
        table[TIME_ON_1_SP] = 1 / (table[SP_IN_HOUR] + 1e-8)
        table[WEEK_ON_TASK] = table[SP_IN_HOUR] * 40
        table[DAY_ON_1_TASK] = table[SP_IN_HOUR] * 8
        table[TIME_ON_1_TASK] = table[TIME_ON_1_SP] * table[TASK_IN_SP]
        table[SUM_HOURS] = [sum_spent_hour] + [
            None for _ in range(table.shape[0] - 1)
        ]
        table[SUM_SP] = [sum_spent_sp] + [
            None for _ in range(table.shape[0] - 1)
        ]
        return table


class TableWithSpAndHourStatisticSprints(models.BaseTableHandler):
    def compute_statistic(
            self,
            key_to_estimated_time: Optional[str] = 'storyPoints',
            key_to_spent_time: Optional[str] = 'spent',
            **kwargs,
    ):
        data = self.data
        df = None
        for i in range(data.shape[0]):
            line = data.iloc[i : i + 1, :].reset_index()
            stat = (
                TableWithSpAndHourStatistic(
                    data.iloc[i : i + 1, :], transform_hours_to_float=False,
                )
                .compute_statistic(key_to_spent_time=key_to_spent_time)
                .drop(labels=[COUNT_TASKS], axis=1)
                .reset_index()
            )
            fast_slow = (
                TableWithSlowAndFastTasks(
                    data.iloc[i : i + 1, :], transform_hours_to_float=False,
                )
                .compute_statistic(
                    intervals=kwargs['intervals'],
                    sp_is_hour=kwargs['sp_is_hour'],
                    estimated=key_to_estimated_time,
                    spent=key_to_spent_time,
                )
                .reset_index()
            )
            line = utils.union_list_value(
                line, ['abcService_name', 'components_name', 'tags'],
            )
            overall_line = pd.concat(
                [
                    line[COLUMNS_FROM_DF],
                    stat[COLUMNS_OVERALL_STAT],
                    fast_slow[COLUMNS_FROM_FAST],
                ],
                sort=True,
                axis=1,
            )
            if df is None:
                df = overall_line
            else:
                df = pd.concat([df, overall_line])
        for col in [
                'abcService_name',
                'assignee_display',
                'components_name',
                'epic_key',
                'goals',
                'project_key',
                'sprint_name',
                'tags',
                'sprint_startDate',
                'sprint_endDate',
                'summary',
                'createdAt',
                'resolvedAt',
        ]:
            df = utils.set_value_to_none(df, col, 'None')
        for col in [
                'spent',
                'spentSp',
                TIME_ON_1_SP,
                DAY_ON_1_TASK,
                'WeekOnOneTaskTypeInSp',
                TIME_ON_1_TASK,
                SP_IN_HOUR,
                'sprint_id',
        ]:
            df.loc[df[col].isna(), col] = -1
        for col in ['sprint_startDate', 'sprint_endDate']:
            df = utils.get_first_element_from_list(df, col)
        for col in ['sprint_startDate', 'sprint_endDate']:
            df = utils.date_to_int(df, col)
        for col in ['createdAt', 'resolvedAt']:
            df = utils.datetime_to_int_shift(df, col)
        return df


class TableWithSlowAndFastTasks(models.BaseTableHandler):
    def compute_statistic(
            self,
            estimated: Optional[str] = 'storyPoints',
            spent: Optional[str] = 'spent',
            **kwargs,
    ):
        data = self.data
        url = 'https://st.yandex-team.ru/'
        needed_columns = ['assignee_display', 'key', estimated, spent]
        dataframe = None
        for sprint in data['sprint_id'].unique():
            index_of_sprint = data['sprint_id'] == sprint
            sprint_data = data[index_of_sprint].copy()
            for key in sprint_data.storyPoints.unique():
                index_key = sprint_data.storyPoints == key
                sp_data = sprint_data[needed_columns][index_key]
                sp_data.loc[:, 'key'] = sp_data.key.apply(lambda x: url + x)
                sp_data.insert(4, 'type', -1, True)
                sp_data.insert(5, 'type_15', -1, True)
                if key in kwargs['intervals'] and not kwargs['sp_is_hour']:
                    faster_15 = (
                        sp_data[spent] < kwargs['intervals'][key][0] * 0.75
                    )
                    faster = sp_data[spent] < kwargs['intervals'][key][0]
                elif kwargs['sp_is_hour']:
                    faster_15 = sp_data[spent] < sp_data[estimated] * 0.75
                    faster = sp_data[spent] < sp_data[estimated]
                else:
                    faster = faster_15 = None
                if faster is not None and faster_15 is not None:
                    sp_data.loc[faster, 'type'] = 2
                    sp_data.loc[faster_15, 'type_15'] = 2
                if key in kwargs['intervals'] and not kwargs['sp_is_hour']:
                    slow_15 = (
                        sp_data[spent] > kwargs['intervals'][key][1] * 1.15
                    )
                    slow = sp_data[spent] > kwargs['intervals'][key][1] * 1.15
                elif kwargs['sp_is_hour']:
                    slow_15 = sp_data[spent] > sp_data[estimated] * 1.15
                    slow = sp_data[spent] > sp_data[estimated]
                else:
                    slow_15 = slow = None
                if slow is not None and slow_15 is not None:
                    sp_data.loc[slow_15, 'type_15'] = 0
                    sp_data.loc[slow, 'type'] = 0
                if key in kwargs['intervals'] and not kwargs['sp_is_hour']:
                    normal_15 = (
                        sp_data[spent] >= kwargs['intervals'][key][0] * 0.75
                    ) & (sp_data[spent] <= kwargs['intervals'][key][1] * 1.15)
                    normal = (
                        sp_data[spent] >= kwargs['intervals'][key][0]
                    ) & (sp_data[spent] <= kwargs['intervals'][key][1])
                elif kwargs['sp_is_hour']:
                    normal_15 = (
                        sp_data[spent] >= sp_data[estimated] * 0.75
                    ) & (sp_data[spent] <= sp_data[estimated] * 1.15)
                    normal = normal_15
                else:
                    normal = normal_15 = None
                if normal is not None and normal_15 is not None:
                    sp_data.loc[normal_15, 'type_15'] = 1
                    sp_data.loc[normal, 'type'] = 1
                dataframe = pd.concat(
                    [dataframe, sp_data.sort_values(by=['assignee_display'])],
                )
        if dataframe is None:
            dataframe = self.data
            dataframe.loc[:, 'key'] = dataframe.key.apply(lambda x: url + x)
            dataframe['type'] = -1
            dataframe['type_15'] = -1
            dataframe = dataframe[needed_columns + ['type', 'type_15']]
        return dataframe
