# coding=utf-8

from typing import Any, Dict, List, Optional, Union
from datetime import datetime
import pytz

import pandas as pd


def get_attr(obj: object, attr: str) -> Any:
    """
     Function for get attribute object and ignore AttributeError
    Args:
        obj: any object
        attr: attribute of obj
    Returns:
        attribute of object or None if it doesn't exist
    """
    try:
        return getattr(obj, attr)
    except AttributeError:
        return None


def string2time(datetime: Union[None, str]) -> Union[None, float]:
    """
    Function for transform spent datetime into spent hours
    Args:
        datetime: datetime in format -
            <int>Y<int>M<int>W<int>DT<int>H<int>M<int>S

    Returns:
        spent hours
    """
    if datetime is None:
        return None
    date_and_time = datetime.split('T')
    if len(date_and_time) == 1:
        return string_date2hours(date_and_time[0])
    elif len(date_and_time) == 2:
        return string_date2hours(date_and_time[0]) + string_time2hours(
            date_and_time[1],
        )
    else:
        raise ValueError(
            'Strange input value into string2time - {}'.format(datetime),
        )


def str_date_or_time2time(
        dict_transformation: Dict[str, Union[float, int]], string: str,
) -> Union[float, int]:
    """
    Function for transform spent datetime into spent hours with dict
    Args:
        dict_transformation: dict with key - parameter in datetime (year,
            month, week, day and etc); value - count of hours of parameter
        string: input datetime

    Returns:

    """
    resulting_hour = 0
    numbers = []
    for char in string:
        if char.isdigit():
            numbers.append(char)
        elif char in dict_transformation:
            resulting_hour += int(''.join(numbers)) * dict_transformation[char]
            numbers = []
    return resulting_hour


def string_date2hours(date):
    date_to_value = {'Y': 250 * 8, 'M': 21 * 8, 'W': 5 * 8, 'D': 8}
    return str_date_or_time2time(date_to_value, date)


def string_time2hours(time):
    time_to_value = {'H': 1, 'M': 1.0 / 60}
    return str_date_or_time2time(time_to_value, time)


def set_none_or_parameter(parameter):
    return None if parameter == 'None' else parameter


def prepare_date(string: str):
    if string:
        date = string.split('-')
        if len(date) != 3:
            raise ValueError('The date is set incorrectly.')
        return tuple(map(int, date))
    else:
        return None


def union_list_value(df, columns):
    for col in columns:
        df[col] = df[col].apply(lambda x: '&'.join(sorted(x)))
    return df


def set_value_to_none(df, column, value):
    df[column] = df[column].apply(lambda x: x if x else value)
    return df


def get_first_element_from_list(df, column):
    df[column] = df[column].apply(lambda x: x[0] if isinstance(x, list) else x)
    return df


def date_to_int(df, column):
    df[column] = df[column].apply(
        lambda x: (
            int(datetime.strptime(x, '%Y-%m-%d').timestamp()) // 86400
            if x != 'None'
            else None
        ),
    )
    return df


def datetime_to_int_shift(df, column):
    msk_tz = pytz.timezone('Europe/Moscow')
    df[column] = df[column].apply(
        lambda x: (
            int(
                msk_tz.normalize(
                    datetime.strptime(x, '%Y-%m-%dT%H:%M:%S.%f%z')
                    .replace(tzinfo=pytz.utc)
                    .astimezone(msk_tz),
                ).timestamp(),
            )
            if x != 'None'
            else None
        ),
    )
    return df


def prepare_dataframe(
        dataframe: pd.DataFrame,
        columns_to_int: List[str],
        columns_to_float: List[str],
        acceptable_tags: Optional[List[str]] = None,
        invalid_tags: Optional[List[str]] = None,
) -> pd.DataFrame:
    if dataframe.shape[0] == 0:
        return dataframe
    for col in columns_to_int:
        dataframe[col] = dataframe[col].apply(
            lambda x: int(x) if x is not None else None,
        )
    for col in columns_to_float:
        dataframe[col] = dataframe[col].apply(
            lambda x: float(x) if x is not None else None,
        )
    dataframe['iterationCount'] = dataframe['sprint_name'].apply(
        lambda x: 1 if len(x) == 1 else 0,
    )
    dataframe['resolution_id'] = dataframe['resolution_id'].apply(
        lambda x: 1 if x == 1 else 0,
    )
    for col in ['sprint_id', 'sprint_name']:
        dataframe[col] = dataframe[col].apply(lambda x: x[-1] if x else None)
    dataframe['goals'] = dataframe['goals'].apply(
        lambda x: x[0].get('title', None) if x else None,
    )
    if acceptable_tags:
        dataframe['tags'] = dataframe['tags'].apply(
            lambda x: [_ for _ in x or [] if _ in acceptable_tags],
        )
    elif invalid_tags:
        dataframe['tags'] = dataframe['tags'].apply(
            lambda x: [_ for _ in x or [] if _ not in invalid_tags],
        )
    return dataframe
