import re
import logging

from enum import Enum
from collections import namedtuple


class TaskTypeLink(Enum):
    """
    Class of link types for startrek task
    """
    Block = 1  # блокирует
    Depend = 2  # зависит
    Relate = 3  # связано
    Duplicate = 4  # дублирует
    Other = 5  # все типы https://st-api.yandex-team.ru/v2/linktypes


LinkedTask = namedtuple('LinkedTask', ['name', 'queue', 'type', 'type_link'])


class ListLinkedTasks:
    """
    Class of linked to task tickets
    """
    def __init__(self, tasks):
        """
        Init function for creating list of linked task
        Args:
            tasks (list of LinkedTask):
        """
        self._tasks = tasks

    @classmethod
    def from_json(cls, json, startrek_client):
        """
        Function for create ListLinkedTasks from json
        Args:
            json (dict): json response with linked task
            startrek_client: api client of startrek
        Returns:
            instance of ListLinkedTasks
        """
        queue_pattern = r'(.+)-[0-9]+'
        tasks = []
        for task in json:
            name = task['object']['key']
            queue = re.search(queue_pattern, name).group(1)
            type_task = cls.get_task_type(startrek_client, name)
            type_link = cls.get_link_type(task)

            if type_link is not TaskTypeLink.Other:
                tasks.append(LinkedTask(name, queue, type_task, type_link))
        return cls(tasks)

    @staticmethod
    def get_link_type(task):
        """
        Function for get link type from json
        Args:
            task (dict): json data
        Returns:
            TaskTypeLink
        """
        type_name = task['type']['id']
        direction = task['direction']
        if type_name == 'depends' and direction == 'inward':
            return TaskTypeLink.Block
        elif type_name == 'depends' and direction == 'outward':
            return TaskTypeLink.Depend
        elif type_name == 'relates':
            return TaskTypeLink.Relate
        elif type_name == 'duplicates':
            return TaskTypeLink.Duplicate
        else:
            return TaskTypeLink.Other

    @staticmethod
    def get_task_type(startrek_client, task_name):
        """
        Function for get task type
        Args:
            task_name (str): task name
            startrek_client: api client of startrek
        Returns:
            task type (str)
        """
        data = startrek_client.get_task_data(task_name)
        return data['type']['key']

    def __iter__(self):
        return self._tasks.__iter__()


class CounterLinks:
    """
    Class for count linked tasks
    """
    def __init__(self, startrek_client, dev_tasks_types, dev_tasks_queues, field):
        """
        Init function
        Args:
            startrek_client (client): for startrek api
            dev_tasks_types (list of str): with possible dev ticket type
            dev_tasks_queues (list of str): with possible queues for
                dev tickets
            field: name field in ticket for writing count bug reports
        """
        self._startrek_client = startrek_client
        self._dev_types = dev_tasks_types
        self._dev_queues = dev_tasks_queues
        self._field = field

    @staticmethod
    def _count_bug_reports(linked_tasks):
        """
        Function for counter task from queue - EDABUGREPORT
        Args:
            linked_tasks (list of LinkedTask): list of linked task
        Returns:
            count (int)
        """
        return sum(1 for task in linked_tasks if task.queue == 'EDABUGREPORT')

    def _find_dev_task(self, linked_tasks):
        """
        Function for find develop task
        Args:
            linked_tasks (list of LinkedTask):
        Returns:
            list of str with dev tasks names
        """
        dev_tasks = []
        for task in linked_tasks:
            if (
                task.type_link in [TaskTypeLink.Block, TaskTypeLink.Depend]
                and task.queue in self._dev_queues
                and task.type in self._dev_types
            ):
                dev_tasks.append(task.name)
        return dev_tasks

    def count_and_write_links(self, query):
        """
        Function for count tasks links and write it to dev ticket
        Args:
            query (str): query to queue EDADUTY
        """
        task_names = self._startrek_client.get_names(query)
        for task_name in task_names:
            linked_tasks = ListLinkedTasks.from_json(
                self._startrek_client.get_links(task_name),
                self._startrek_client
            )
            count_reports = self._count_bug_reports(linked_tasks)
            logging.info(
                'Founded {} bug reports for {}'.format(
                    count_reports, task_name
                )
            )
            dev_tasks_names = self._find_dev_task(linked_tasks)
            if len(dev_tasks_names) == 0:
                logging.info(
                    'For task {} in \'Зависит от\', \'Блокирует\' not founded'
                    ' tickets with writed queues and task types'.format(
                        task_name
                    )
                )
            for dev_task_name in dev_tasks_names:
                self._startrek_client.set_field(
                    dev_task_name,
                    self._field,
                    count_reports
                )
