import logging

import requests


def _send_request(
    method, url, oauth_token, headers=None, params=None, data=None
):
    """
    Function for sending requests
    Args:
        method (str): type of request 'POST', 'GET' or 'PATCH' and etc
        url (str): URL for request
        oauth_token (str): OAuth token
        headers (dict): headers for request
        params (dict): params for request
        data (dict or string): data for request
    Returns:
        tuple of
            json data (dict or None) and
            response status code (int)
    """
    kwargs = {
        'method': method,
        'url': url,
        'headers': {
            'Authorization': 'OAuth {}'.format(oauth_token)
        }
    }
    if headers is not None and isinstance(headers, dict):
        kwargs['headers'].update(headers)
    if params is not None and isinstance(params, dict):
        kwargs['params'] = params
    if data is not None:
        kwargs['data'] = data

    response = requests.request(**kwargs)

    if response.status_code != 200:
        return None, response.status_code
    return response.json(), response.status_code


class Startrek:
    def __init__(self, oauth):
        self._oauth = oauth
        self._base_url = 'https://st-api.yandex-team.ru'

    def get_links(self, task_name):
        """
        Function for get links for task
        Args:
            task_name (str): name of task
        """
        data, status_code = _send_request(
            method='GET',
            url=self._base_url + '/v2/issues/{}/links'.format(task_name),
            oauth_token=self._oauth
        )
        logging.info('Request ended with status {} for task {} links'.format(
            status_code, task_name
        ))
        if data is None:
            raise ValueError(
                'Can\'t get linked tickets for task {}'.format(task_name)
            )
        return data

    def get_names(self, query):
        """
        Function for get task name by query
        Args:
            query (str): query to startrek
        Returns:
            list of task name
        """
        params = {'query': query, 'perPage': '100'}
        data, status_code = _send_request(
            method='GET',
            url=self._base_url + '/v2/issues',
            oauth_token=self._oauth,
            params=params
        )
        logging.info('Request ended with status {} for query {}'.format(
            status_code, query
        ))
        if data is None:
            raise ValueError(
                'Can\'t get name ticket by query - {}'.format(query)
            )
        if status_code == 200:
            data = [task['key'] for task in data]
        return data

    def set_field(self, task_name, field, value):
        """
        Function for setting value in field
        Args:
            task_name (str): task name
            field (str): field for setting
            value (int, float or str): value
        """
        headers = {'Content-Type': 'application/json'}
        logging.info('Setting {} into {}'.format(
            value, field
        ))
        _, status_code = _send_request(
            method='PATCH',
            oauth_token=self._oauth,
            url='https://st-api.yandex-team.ru'
                '/v2/issues/{}'.format(task_name),
            headers=headers,
            data='{{"{}": "{}"}}'.format(field, value)
        )
        logging.info(
            'Request ended with status {} for setting {} into {} of {}'.format(
                status_code, value, field, task_name
            )
        )

    def get_task_data(self, task_name):
        """
        Function for get information about one ticket
        Args:
            task_name (str): task name in startrek
        Returns:
            dict - json with information
        """
        data, status_code = _send_request(
            method='GET',
            url=self._base_url + '/v2/issues/{}'.format(task_name),
            oauth_token=self._oauth
        )
        logging.info(
            'Request ended with status {} for data about task {}'.format(
                status_code, task_name
            )
        )
        if data is None:
            raise ValueError(
                'Can\'t get data about {} task'.format(task_name)
            )
        return data
