# coding=utf-8

import logging

from typing import Dict, List, Optional, Tuple, Union

import requests

from . import utils
from . import exceptions

NUMBERS = Union[float, int]
JSON_LIKE = Dict[str, Union[NUMBERS, str]]


def _send_request(
    method: str, url: str, oauth_token: str,
    headers: Optional[JSON_LIKE] = None,
    params: Optional[JSON_LIKE] = None,
    data: Optional[Union[str, JSON_LIKE]] = None,
) -> Tuple[Union[None, JSON_LIKE, List[JSON_LIKE]], int]:
    """
    Function for sending requests
    Args:
        oauth_token: OAuth token
        method: type of request 'POST', 'GET' or 'PATCH' and etc
        url: URL for request
        headers: headers for request
        params: params for request
        data: data for request
    Returns:
        json data and response status code
    """
    kwargs = {
        'headers': {
            'Authorization': 'OAuth {}'.format(oauth_token)
        },
        'method': method,
        'url': url,
    }
    if headers is not None:
        kwargs['headers'].update(headers)
    if params is not None:
        kwargs['params'] = params
    if data is not None:
        kwargs['data'] = data

    response = requests.request(**kwargs)

    if response.status_code != 200:
        return None, response.status_code
    return response.json(), response.status_code


class StartrekClient:
    """
    Class with some requests to startrek
    """
    _base_url = 'https://st-api.yandex-team.ru/v2/issues'

    def __init__(self, oauth: str):
        """
        Init function
        Args:
            oauth: OAuth token for using API startrek
        """
        self._oauth_token = oauth

    def download_data(self, query: str) -> Optional[List[JSON_LIKE]]:
        """
        Function for getting data from startrek
        Args:
            query: Query to startrek
        Returns:
            tickets
        """
        result = []
        params = {'query': query, 'perPage': '100'}
        for page in range(1, 100):
            params['page'] = page
            response, status = _send_request(
                method='GET',
                oauth_token=self._oauth_token,
                params=params,
                url=self._base_url,
            )
            if not response:
                break
            logging.info(f'Response status for query data is {status}')
            if status != 200:
                exit(-1)
            result.extend(response)
        return result

    def set_priority(
        self,
        task_name: str,
        weight: NUMBERS,
        mapping_weights_to_priority: Dict[Tuple[NUMBERS], str],
    ) -> None:
        """
        Function for setting priorities to tasks
        Args:
            task_name: task name
            weight: task weight
                priority. Key - tuple of left and right edge, value - priority
                to this interval
            mapping_weights_to_priority: mapping interval of weights to
                priority
        """
        headers = {'Content-Type': 'application/json'}
        logging.info(f'Weight is {weight} for {task_name}')
        priority = None
        for key in mapping_weights_to_priority:
            if key[0] <= weight <= key[1]:
                priority = mapping_weights_to_priority[key]
        if priority is not None:
            _, status_code = _send_request(
                data=f'{{"priority": "{priority}"}}',
                headers=headers,
                method='PATCH',
                oauth_token=self._oauth_token,
                url=self._base_url + f'/{task_name}',
            )
            logging.info(
                f'Response updating priority for {task_name} with '
                f'setting status "{priority}" has status code {status_code}'
            )

    def add_tag(self, task_name: str, tag: str) -> None:
        """
        Function for setting value in field
        Args:
            task_name: task name
            tag: adding tag value
        """
        headers = {'Content-Type': 'application/json'}
        logging.info(f'Setting {tag} into tags')
        _, status_code = _send_request(
            data=f'{{"tags": {{"add":"{tag}"}}}}',
            headers=headers,
            method='PATCH',
            oauth_token=self._oauth_token,
            url=self._base_url + f'/{task_name}',
        )
        logging.info(
            f'Request ended with status {status_code} for adding {tag} '
            f'into tags of {task_name}'
        )

    def set_field(
        self, task_name: str, field: str, value: Union[NUMBERS, str]
    ) -> None:
        """
        Function for setting value in field
        Args:
            task_name: task name
            field: weight setting field
            value: value
        """
        headers = {'Content-Type': 'application/json'}
        logging.info(f'Setting {value} into {field}')
        _, status_code = _send_request(
            data=f'{{"{field}": "{value}"}}',
            headers=headers,
            method='PATCH',
            oauth_token=self._oauth_token,
            url=self._base_url + f'/{task_name}',
        )
        logging.info(
            f'Request ended with status {status_code} for setting {value}'
            f' into {field} of {task_name}'
        )


class WeightCalculator:
    """
    Class formula for automatic computing weights
    """

    def __init__(self, formula: str):
        """
        Init function
        Args:
            formula: multiline string with formula
        """
        self._formula = utils.preprocessing_formula(formula)

    def compute_weight(self, task: JSON_LIKE) -> Optional[NUMBERS]:
        """
        Method for completing values in formula for task
        Args:
            task: data about one task from client
        Returns:
            weight of task
        """
        weight = 0
        for operand, value, condition, path in self._formula:
            if path.strip() == 'THIS':
                path_value = weight
            else:
                path_value = utils.get_attr_of_data(path, task)
            if path_value is None:
                # path or value as path in data not exist
                raise exceptions.FormulaError(
                    'For task {} not founded path {}'.format(
                        task['key'],
                        path
                    )
                )
            if '{}' in condition:
                condition = condition.format(
                    utils.preprocessing_string(
                        repr(path_value)
                    )
                )
            condition = eval(condition)
            if condition:
                weight = utils.update_weight(weight, operand, value)
        logging.info(
            'Weight is {} for task {}'.format(
                weight, task['key']
            )
        )
        return weight


class UpdatingFields:
    """
    Class check tag and save tag and field value
    """

    def __init__(self, field_for_weight, force_update, tag_for_check):
        """
        Init function
        Args:
            field_for_weight (str): field for setting weight
            force_update (bool): is force update tags and priority
            tag_for_check (str): tag checking value
        """
        self.field_for_weight = field_for_weight
        self.force_update = force_update
        self.tag_for_check = tag_for_check

    def check_tag(self, task):
        """
        Function of checking tag
        Args:
            task (dict): json like data about ticket
        Returns:
            bool or raise TagError
        """
        if (
            not self.force_update  # throw TagError if force update off
            and (
            not self.tag_for_check  # if tag_for_check is empty
            or (  # or
                'tags' in task and  # check 'tags' in json response
                self.tag_for_check in task['tags']  # tag_for_check in tags
            )
        )
        ):
            if not self.tag_for_check:
                raise exceptions.TagError(
                    'Task {} skipped because you set force_update = False '
                    'and tag_for_check is empty.'.format(
                        task['key']
                    )
                )
            else:
                raise exceptions.TagError(
                    'In task {} founded tag {}. Task skipped'.format(
                        task['key'],
                        self.tag_for_check
                    )
                )
        elif self.tag_for_check:
            return True
        return False


class AutoUpdaterPriority:
    """
    Class for update tasks priority
    """

    def __init__(
        self, startrek_client, calculator, fields, mapping_weight_priority
    ):
        """
        Init function
        Args:
            startrek_client (instance of StartrekClient): startrek client
            calculator (instance od WeightCalculator): calculator
            fields (instance of UpdatingFields): check tag and save fields
        """
        self.client = startrek_client
        self.calculator = calculator
        self.fields = fields
        self.mapping_weight_priority = mapping_weight_priority

    def update_priority(self, query):
        tasks = self.client.download_data(query)
        assert len(tasks), 'Не найдены тикеты, проверьте запрос и доступ робота'
        ' @ robot-pmo-automation'
        for task in tasks:
            try:
                result_of_tag_check = self.fields.check_tag(task)
                weight = self.calculator.compute_weight(task)
            except (exceptions.FormulaError, exceptions.TagError) as error:
                logging.info(error.args[0])
                continue
            # set weight to field
            if self.fields.field_for_weight:
                self.client.set_field(
                    task['key'], self.fields.field_for_weight, weight
                )
            # set tag if it needed
            if result_of_tag_check:
                self.client.add_tag(task['key'], self.fields.tag_for_check)
            self.client.set_priority(task['key'], weight, self.mapping_weight_priority)
