# coding=utf-8

import logging

from typing import Optional, Tuple

import jmespath


# function for preprocess data


def prepare_tuple_interval(interval_string: str) -> Tuple[float, ...]:
    """
    Function for prepare edge interval from string
    Args:
        interval_string:  left and right edge separated whitespace
    Returns:
        tuple of int - left and right edge
    """
    return tuple(
        sorted(
            map(
                lambda x: float(x.strip()),
                interval_string.split(' ')
            )
        )
    )


def preprocessing_string(value: str, delete_comment: Optional[bool] = False):
    """
    Function for preprocessing string. It cast unicode to str if need and
    delete comment
    Args:
        value (str or unicode): input value
        delete_comment (bool): if True delete comment from string starting
            with symbol #
    Returns:
        str - input value
    """
    string = value.strip()
    if delete_comment:
        string = string.split('#', 1)[0].strip()
    return string


def preprocessing_formula(formula):
    """
    Function for preprocessing formula from string
    Args:
        formula (str): multiline string with formulas
    Returns:
        list of tuple
         with 4 value:
            1. Operand (str)
            2. Value or JsonPath (float or str)
            3. Logical condition (str)
            4. JsonPath (str)
            after # you can write comment
    """
    operations = []
    formula = preprocessing_string(formula)
    for line in formula.splitlines():
        operand, value, condition, json_path = map(
            lambda x: preprocessing_string(x, delete_comment=True).strip(),
            line.split('|')
        )
        try:
            value = float(value)
        except ValueError:
            pass  # value is JsonPath
        operations.append(
            (
                operand,
                value,
                condition,
                json_path
            )
        )
    return operations


def get_attr_of_data(json_path, data):
    """
    Function for getting data by attribute from json
    Args:
        json_path: (str) - JsonPath like string
        data: (dict) - json data
    Returns:
        data from JsonPath in json data - dict or None if path in json
            doesn't exist
    """
    value = jmespath.search(json_path.strip(), data)
    return value


# evaluation weights


def set_values_to_formula(formula, data):
    """
    Function for set values from data to formula
    Args:
        formula (list of tuple): preprocessed formula
        data (dict): json data
    Returns:
        same as output from function preprocessing_and_prepare_formula
        return [(False, path, None)] if some JsonPath is not exist
    """
    completed_formulas = []
    for operand, value, condition, path in formula:
        path_value = get_attr_of_data(path, data)
        if isinstance(value, str):  # set value if it JsonPath
            value = get_attr_of_data(value, data)
        if path_value is None or value is None:
            # path or value as path in data not exist
            return [
                (False, path, None)
            ]
        if '{}' in condition:
            condition = condition.format(repr(path_value))
        condition = eval(condition)
        if condition:
            completed_formulas.append(
                (path_value, operand, value)
            )
    return completed_formulas


def update_weight(weight, operand, value):
    """
    Function for update weight value
    Args:
        weight (float or int): current weight value
        operand (str): math operand one of [+, -, /, *, **]
        value (str): value changing weight
    Returns:
        float or int - updated weight
    """
    operand = operand.strip()
    assert operand in ['*', '**', '+', '/', '-'], 'Not right operand'
    if weight == 0 and operand == '*':
        weight = 1
    return eval('{}{}{}'.format(weight, operand, value))


def compute_weights(data_query, list_of_operations):
    """
    Function for compute weight to tasks
    Args:
        data_query (dict):
        list_of_operations (object):
    Returns:
        tuple of two list - names of tasks (list of str)
        and weights of tasks (list of float or int)
    """
    name_of_tasks, weight_of_tasks = [], []
    for task in data_query:
        formulas_for_task = set_values_to_formula(
            list_of_operations,
            task
        )
        if len(formulas_for_task) == 1 and not formulas_for_task[0][0]:
            # skip task if key doesn't exist
            logging.info(
                'For task - {} key - {} not founded'.format(
                    task['key'],
                    formulas_for_task[0][1]  # path
                )
            )
            continue
        weight = 0
        for path_value, operand, value in formulas_for_task:
            weight = update_weight(weight, operand, value)
        logging.info(
            'Weight is {} for task {}'.format(
                weight, name_of_tasks
            )
        )
        name_of_tasks.append(task['key'])
        weight_of_tasks.append(weight)
    return name_of_tasks, weight_of_tasks
