# -*- coding: utf-8 -*-
import logging
import subprocess

from sandbox import sdk2
from sandbox.projects.education.resources import WorkflowBinary


class HiddenData(str):
    def __repr__(self):
        return '*****'


def run_command_with_livelog(command, env=None):

    logging.info('RUNNING COMMAND: {}'.format(command))

    # temporary change format to just message from binary
    root_logger = logging.getLogger()
    simple_formatter = logging.Formatter('%(message)s')
    prev_formatters = [handler.formatter for handler in root_logger.handlers]
    for handler in root_logger.handlers:
        handler.formatter = simple_formatter

    try:

        proc = subprocess.Popen(
            command,
            stdout=subprocess.PIPE,
            stderr=subprocess.STDOUT,
            bufsize=1,
            env=env
        )

        with proc.stdout:
            for line in iter(proc.stdout.readline, ''):
                logging.warning(line.rstrip())

        proc.wait()

        if proc.returncode != 0:
            message = '{} failed, with status code {}'.format(command, proc.returncode)
            raise RuntimeError(message)

    finally:
        # reset formatter
        for handler, prev_formatter in zip(root_logger.handlers, prev_formatters):
            handler.formatter = prev_formatter


class WorkflowBinaryExecutor(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Group of parameters for binary runner workflow'):
            workflow_binary = sdk2.parameters.Resource(
                'Workflow binary',
                resource_type=WorkflowBinary,
                required=True
            )
            environment = sdk2.parameters.String(
                'Environment',
                default='testing'
            )
            valhalla_oauth_token_name = sdk2.parameters.String(
                'Valhalla OAuth token name',
                default=''
            )

    def on_execute(self):
        binary_path = str(sdk2.ResourceData(self.Parameters.workflow_binary).path)

        command = [
            binary_path,
            '-e', self.Parameters.environment,
            '-t', HiddenData(sdk2.Vault.data(self.Parameters.owner, self.Parameters.valhalla_oauth_token_name)),
        ]

        run_command_with_livelog(command)
