# coding: utf-8

import logging

import os
import shutil

from sandbox import sdk2
import sandbox.common.types.task as ctt

from sandbox.projects.common.nanny.nanny import ReleaseToNannyTask2


task_types = [
    'BUILD_EPSILOND1_BINARY'
]


class BuildEpsilond1(ReleaseToNannyTask2, sdk2.Task):
    class Context(sdk2.Task.Context):
        children = []
        release_parameters = None

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Common') as block_common:
            description = sdk2.parameters.String('Description', default='<no description>')
            path = sdk2.parameters.String('SVN path', default='arcadia:/arc/trunk/arcadia')
            timeout = sdk2.parameters.Float('Timeout (seconds)', default=900)

        with sdk2.parameters.Group('Tasks') as block_tasks:
            BUILD_EPSILOND1_BINARY = sdk2.parameters.Bool(label='BUILD_EPSILOND1_BINARY')

    def get_subtasks(self):
        for task_type in task_types:
            if not getattr(self.Parameters, task_type):
                continue

            task_type = sdk2.Task[task_type]
            task = task_type(task_type.current, description=self.Parameters.description, checkout_arcadia_from_url=self.Parameters.path)
            self.Context.children.append(task.id)
            yield task.enqueue()

    def on_execute(self):
        with self.memoize_stage.create_children:
            tasks = list(self.get_subtasks())
            raise sdk2.WaitTask(
                tasks,
                ctt.Status.Group.FINISH + ctt.Status.Group.BREAK,
                wait_all=True,
                timeout=self.Parameters.timeout,
            )

        failed = set()
        for task_id in self.Context.children:
            # Check statuses.
            task = self.find(id=task_id).first()  # type: sdk2.Task
            logging.info('-> found task for id=%s: %s', task_id, task)

            if task.status != ctt.Status.SUCCESS:
                failed.add(task)
            elif not failed:
                for resource in sdk2.Resource.find(task=task).limit(100):  # type: sdk2.Resource
                    if not resource.releasable:
                        continue

                    logging.info('-> %s: cloning resource: %s', task_id, resource)
                    resource_data = sdk2.ResourceData(resource)
                    resource_path = str(resource_data.path)
                    logging.info('-> %s: %s: resource path: %s', task_id, resource.id, resource_path)

                    clone_dir_path = str(self.path('{}.{}'.format(task_id, resource.id)))
                    os.mkdir(clone_dir_path)

                    if os.path.isdir(resource_path):
                        shutil.copytree(resource_path, os.path.join(clone_dir_path, os.path.basename(resource_path)))
                    elif os.path.isfile(resource_path):
                        shutil.copy2(resource_path, clone_dir_path)
                    else:
                        raise RuntimeError('not a file or directory: {} (task={}, resource={})'.format(resource_path, task_id, resource.id))

                    clone_path = os.path.join(clone_dir_path, os.path.basename(resource_path))
                    logging.info('-> %s: %s: clone path: %s', task_id, resource.id, clone_path)

                    resource_type = type(resource)
                    logging.info('-> %s: %s: resource type: %s', task_id, resource.id, resource_type)
                    # Actually clones resource.
                    sdk2.ResourceData(resource_type(self, resource.description, clone_path))

        if failed:
            raise RuntimeError('failed tasks: {}'.format(', '.join([str(x) for x in failed])))
