import datetime
import os

from sandbox import sdk2
from sandbox.common.types.task import Semaphores
from sandbox.sdk2.helpers import subprocess


class DateParameter(sdk2.parameters.String):
    @classmethod
    def cast(cls, value):
        if value:
            datetime.datetime.strptime(value, '%Y-%m-%d')
        return value


class BuildNaviRoutes(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        navi_routes_executable = sdk2.parameters.Resource(
            'Executable for launching navi routes operations',
            required=True,
        )

        with sdk2.parameters.RadioGroup('Route build mode') as mode:
            mode.values['fast'] = mode.Value(value='Fast')
            mode.values['batch'] = mode.Value(value='Batch', default=True)
            mode.values['batch_history'] = mode.Value(value='Batch history')

        with mode.value['batch_history']:
            start_date = DateParameter('Start date to process', required=True)
            days_to_process = sdk2.parameters.Integer('Days to process', required=True, default=1)

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_enqueue(self):
        if self.Parameters.mode != 'batch_history':
            self.Requirements.semaphores = Semaphores(
                acquires=[Semaphores.Acquire(name='extmaps_navi_routes_{mode}'.format(mode=self.Parameters.mode))],
            )
        return super(BuildNaviRoutes, self).on_enqueue()

    def on_execute(self):
        os.environ['YT_TOKEN'] = sdk2.Vault.data('robot_extdata_yt_token')
        os.environ['YQL_TOKEN'] = sdk2.Vault.data('EXTDATA', 'YQL_TOKEN')

        binary = sdk2.ResourceData(self.Parameters.navi_routes_executable)

        with sdk2.helpers.ProcessLog(self, logger='navi_routes') as process_log:
            command = [str(binary.path), '--mode', self.Parameters.mode.replace('_', '-'), '--once']
            if self.Parameters.mode == 'batch_history':
                command.extend([
                    '--start-date',
                    self.Parameters.start_date,
                    '--days-to-process',
                    str(self.Parameters.days_to_process),
                ])
            command.append('--config=hahn')
            subprocess.check_call(command, stdout=process_log.stdout, stderr=subprocess.STDOUT)
