import logging
import os

from sandbox.projects.factorization.tools.params import FactorizationLoadYtExecutable
from sandbox.projects.factorization.tools.params import FactorizationAlsColumnDescriptor
from sandbox.projects.factorization.tools.params import FactorizationAlsData
from sandbox.projects.factorization.tools.params import FactorizationAlsModel

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp


class FactorizationDownload(sdk2.Task):
    ''' Runner tool for load yt [download mode] '''

    class Parameters(sdk2.Parameters):
        load_yt_executable = sdk2.parameters.Resource(
            'Load YT executable',
            resource_type=FactorizationLoadYtExecutable,
        )
        yt_proxy = sdk2.parameters.String('YT proxy', default='hahn.yt.yandex.net')
        yt_token_secret_name = sdk2.parameters.String('YT token vault secret', required=True)

        datai = sdk2.parameters.String('Input data table')

        modeli = sdk2.parameters.String('Input model table')
        lkeysi = sdk2.parameters.String('Input left keys table')
        rkeysi = sdk2.parameters.String('Input right keys table')

        column_descriptor = sdk2.parameters.Resource('Column descriptor', resource_type=FactorizationAlsColumnDescriptor)

        predictions = sdk2.parameters.Bool("Extract predictions")
        features = sdk2.parameters.Bool("Extract features")
        with features.value[True]:
            feature_prefix = sdk2.parameters.String("Feature prefix")

    def on_execute(self):
        yt_token = sdk2.Vault.data(self.Parameters.yt_token_secret_name)
        assert yt_token is not None
        os.environ['MR_RUNTIME'] = 'YT'
        os.environ['YT_TOKEN'] = yt_token

        command = [
            str(sdk2.ResourceData(self.Parameters.load_yt_executable).path),
            '--server', self.Parameters.yt_proxy,
        ]

        if self.Parameters.datai:
            datao_resource = FactorizationAlsData(self, self.Parameters.description + ' downloaded data', 'data')
            datao_resource_data = sdk2.ResourceData(datao_resource)

            command += [
                '--datai', self.Parameters.datai,
                '--datao', str(datao_resource_data.path),
                '--cd', str(sdk2.ResourceData(self.Parameters.column_descriptor).path),
            ]
        if self.Parameters.modeli:
            command += [
                '--mi', self.Parameters.modeli,
                '--mo', './model',
                '--lki', self.Parameters.lkeysi,
                '--lko', './lkeys',
                '--rki', self.Parameters.rkeysi,
                '--rko', './rkeys',
            ]

        if self.Parameters.features:
            command += ['--features', '--fprefix', self.Parameters.feature_prefix]

        if self.Parameters.predictions:
            command += ['--predictions']

        logging.info('Run command: {}'.format(' '.join(command)))
        with sdk2.helpers.ProcessLog(self, logger='load_yt_executor') as l:
            sp.check_call(command, stdout=l.stdout, stderr=l.stderr)

        if self.Parameters.modeli:
            modelo_resource = FactorizationAlsModel(self, self.Parameters.description + ' downloaded model', 'modelo')
            modelo_resource_data = sdk2.ResourceData(modelo_resource)

            pack_command = ['tar', '-czvf', str(modelo_resource_data.path), './model', './lkeys', './rkeys']
            logging.info('Run command: {}'.format(' '.join(pack_command)))
            with sdk2.helpers.ProcessLog(self, logger='pack_executor') as l:
                sp.check_call(pack_command, stdout=l.stdout, stderr=l.stderr)

            modelo_resource_data.ready()

        if self.Parameters.datai:
            datao_resource_data.ready()
