import logging
import os

from sandbox.projects.factorization.tools.params import FactorizationLoadYtExecutable
from sandbox.projects.factorization.tools.params import FactorizationAlsColumnDescriptor
from sandbox.projects.factorization.tools.params import FactorizationAlsData
from sandbox.projects.factorization.tools.params import FactorizationAlsModel

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp


class FactorizationUpload(sdk2.Task):
    ''' Runner tool for load yt [upload mode] '''

    class Parameters(sdk2.Parameters):
        load_yt_executable = sdk2.parameters.Resource(
            'Load YT executable',
            resource_type=FactorizationLoadYtExecutable,
        )
        yt_proxy = sdk2.parameters.String('YT proxy', default='hahn.yt.yandex.net')
        yt_token_secret_name = sdk2.parameters.String('YT token vault secret', required=True)

        datai = sdk2.parameters.Resource('Input data', resource_type=FactorizationAlsData)
        input_model = sdk2.parameters.Resource('Input model', resource_type=FactorizationAlsModel)
        column_descriptor = sdk2.parameters.Resource('Column descriptor', resource_type=FactorizationAlsColumnDescriptor)

        datao = sdk2.parameters.String('Loaded data table')

        modelo = sdk2.parameters.String('Output model table')
        lkeyso = sdk2.parameters.String('Output left keys table')
        rkeyso = sdk2.parameters.String('Output right keys table')

        predictions = sdk2.parameters.Bool("Extract predictions")
        features = sdk2.parameters.Bool("Extract features")
        with features.value[True]:
            feature_prefix = sdk2.parameters.String("Feature prefix")

    def on_execute(self):
        yt_token = sdk2.Vault.data(self.Parameters.yt_token_secret_name)
        assert yt_token is not None
        os.environ['MR_RUNTIME'] = 'YT'
        os.environ['YT_TOKEN'] = yt_token

        if self.Parameters.input_model:
            unpack_command = ['tar', '-xzvf', str(sdk2.ResourceData(self.Parameters.input_model).path)]

            logging.info('Run command: {}'.format(' '.join(unpack_command)))
            with sdk2.helpers.ProcessLog(self, logger='unpack_executor') as l:
                sp.check_call(unpack_command, stdout=l.stdout, stderr=l.stderr)

        command = [
            str(sdk2.ResourceData(self.Parameters.load_yt_executable).path),
            '--up',
            '--server', self.Parameters.yt_proxy,
        ]

        if self.Parameters.datai:
            command += [
                '--datai', str(sdk2.ResourceData(self.Parameters.datai).path),
                '--datao', self.Parameters.datao,
                '--cd', str(sdk2.ResourceData(self.Parameters.column_descriptor).path),
            ]
        if self.Parameters.input_model:
            if self.Parameters.modelo:
                command += [
                    '--mi', './model',
                    '--mo', self.Parameters.modelo,
                ]
            if self.Parameters.lkeyso:
                command += [
                    '--lki', './lkeys',
                    '--lko', self.Parameters.lkeyso,
                ]
            if self.Parameters.rkeyso:
                command += [
                    '--rki', './rkeys',
                    '--rko', self.Parameters.rkeyso,
                ]

        if self.Parameters.features:
            command += ['--features', '--fprefix', self.Parameters.feature_prefix]

        if self.Parameters.predictions:
            command += ['--predictions']

        logging.info('Run command: {}'.format(' '.join(command)))
        with sdk2.helpers.ProcessLog(self, logger='load_yt_executor') as l:
            sp.check_call(command, stdout=l.stdout, stderr=l.stderr)
