# -*- coding: utf-8 -*-$

import os
import re
import time

from sandbox import sdk2
from sandbox.projects.common import error_handlers as eh
from sandbox.sandboxsdk.channel import channel


NUM_TOP_RESOURCES_TO_REVISE = 100
NUM_DAYS_TO_KEEP_INF = 30
NUM_DAYS_TO_KEEP_OLD = 30

CREDIBILITY_TRIES_RESOURCE_CONFIG_NAME = 'ya.make'
DEFAULT_CREDIBILITY_TRIES_RESOURCE_CONFIG_PATH = 'search/web/rearrs_upper/rearrange.dynamic/facts'

PER_SOURCE_PRECISION_LIST_RESOURCE_CONFIG_NAME = 'ya.make'
DEFAULT_PER_SOURCE_PRECISION_LIST_RESOURCE_CONFIG_PATH = 'search/web/rearrs_upper/rearrange.dynamic/facts'

FACTSNIP_SCORE_PRECISION_MAP_RESOURCE_CONFIG_NAME = 'ya.make'
DEFAULT_FACTSNIP_SCORE_PRECISION_MAP_RESOURCE_CONFIG_PATH = 'search/web/rearrs_upper/rearrange.dynamic/facts'

PER_SOURCE_PQ_LIST_RESOURCE_CONFIG_NAME = 'ya.make'
DEFAULT_PER_SOURCE_PQ_LIST_RESOURCE_CONFIG_PATH = 'search/web/rearrs_upper/rearrange.dynamic/facts'


def get_newest_resources(resource_type, n, attrs=None):
    return channel.sandbox.list_resources(
        resource_type=resource_type,
        status='READY',
        hidden=False,
        limit=n,
        all_attrs=attrs
    )


def update_resource_in_arcadia(
    new_resource_id,
    arcadia_config_path,
    arcadia_config_filename,
    author,
    trg_line_pattern,
    old_ttl,
    num_days_to_keep_inf,
    resource_type
):
    svn_url = sdk2.svn.Arcadia.trunk_url(arcadia_config_path)
    svn_dir = sdk2.svn.Arcadia.get_arcadia_src_dir(svn_url)
    eh.verify(
        svn_dir,
        'Commit directory set by arcadia_config_path parameter should not be None. Path {} may not exist'.format(
            arcadia_config_path
        )
    )
    svn_dir = str(svn_dir)
    # read lines from the destination file
    lines = []
    with open(os.path.join(svn_dir, arcadia_config_filename), "rt") as _src:
        lines = _src.readlines()
    trg_regex = re.compile(trg_line_pattern.format('[0-9]+') + '.*')
    # and substitute one by a pattern
    old_resource_id = 0
    with open(os.path.join(svn_dir, arcadia_config_filename), "wt") as _trg:
        for line in lines:
            match = trg_regex.search(line)
            if match:
                _trg.write(trg_line_pattern.replace("\\(", "[").replace("\\)", "]").replace("(", "").replace(")", "").replace("[", "(").replace("]", ")").format(new_resource_id) + "\n")
                old_resource_id = match.group(1)
            else:
                _trg.write(line)
    sdk2.svn.Arcadia.commit(
        svn_dir,
        'Another resource ({}) of type ({}) committed. SKIP_CHECK'.format(new_resource_id, resource_type),
        author
    )
    # change old resources' ttl
    for resource in get_newest_resources(resource_type, NUM_TOP_RESOURCES_TO_REVISE):
        if (int(time.time()) - resource.timestamp) > 60 * 60 * 24 * num_days_to_keep_inf and resource.id != old_resource_id:
            sdk2.Resource[resource.id].ttl = old_ttl


class UpdateCredibilityTrieResourceInArcadia(sdk2.Task):
    """ Update resource ID in arcadia"""

    class Parameters(sdk2.Parameters):
        resource_id = sdk2.parameters.Integer('Credibility tries resource ID', required=True)
        arcadia_config_path = sdk2.parameters.ArcadiaUrl(
            'Path to the config with the resource ID',
            required=True,
            default_value=DEFAULT_CREDIBILITY_TRIES_RESOURCE_CONFIG_PATH
        )
        author = sdk2.parameters.String('Author of commit', default='zomb-sandbox-rw', required=True)
        num_days_to_keep_inf = sdk2.parameters.Integer('Number of days to keep resource\'s TTL=INF', default=NUM_DAYS_TO_KEEP_INF)

    def on_execute(self):
        trg_line_pattern = "FROM_SANDBOX\\(({}) OUT credible_hosts.trie credible_masks.trie credible_urls.trie\\)"
        update_resource_in_arcadia(
            self.Parameters.resource_id,
            self.Parameters.arcadia_config_path,
            CREDIBILITY_TRIES_RESOURCE_CONFIG_NAME,
            self.Parameters.author,
            trg_line_pattern,
            NUM_DAYS_TO_KEEP_OLD,
            self.Parameters.num_days_to_keep_inf,
            "FACT_CREDIBILITY_DATABASE",
        )


class UpdatePerSourcePrecisionListResourceInArcadia(sdk2.Task):
    """ Update resource ID in arcadia"""

    class Parameters(sdk2.Parameters):
        resource_id = sdk2.parameters.Integer('Per-fact-source precision list resource ID', required=True)
        arcadia_config_path = sdk2.parameters.ArcadiaUrl(
            'Path to the config with the resource ID',
            required=True,
            default_value=DEFAULT_PER_SOURCE_PRECISION_LIST_RESOURCE_CONFIG_PATH
        )
        author = sdk2.parameters.String('Author of commit', default='zomb-sandbox-rw', required=True)

    def on_execute(self):
        trg_line_pattern = "FROM_SANDBOX\\(FILE ({}) OUT fact_source_precisions.tsv\\)"
        update_resource_in_arcadia(
            self.Parameters.resource_id,
            self.Parameters.arcadia_config_path,
            PER_SOURCE_PRECISION_LIST_RESOURCE_CONFIG_NAME,
            self.Parameters.author,
            trg_line_pattern,
            NUM_DAYS_TO_KEEP_OLD,
            NUM_DAYS_TO_KEEP_INF,
            "FACT_SOURCE_PRECISION_LIST",
        )


class UpdateFactsnipScorePrecisionMapResourceInArcadia(sdk2.Task):
    """ Update resource ID in arcadia"""

    class Parameters(sdk2.Parameters):
        resource_id = sdk2.parameters.Integer('Factsnip score-precision map resource ID', required=True)
        arcadia_config_path = sdk2.parameters.ArcadiaUrl(
            'Path to the config with the resource ID',
            required=True,
            default_value=DEFAULT_FACTSNIP_SCORE_PRECISION_MAP_RESOURCE_CONFIG_PATH
        )
        author = sdk2.parameters.String('Author of commit', default='zomb-sandbox-rw', required=True)

    def on_execute(self):
        trg_line_pattern = "FROM_SANDBOX\\(FILE ({}) OUT fs_segments.tsv\\)"
        update_resource_in_arcadia(
            self.Parameters.resource_id,
            self.Parameters.arcadia_config_path,
            FACTSNIP_SCORE_PRECISION_MAP_RESOURCE_CONFIG_NAME,
            self.Parameters.author,
            trg_line_pattern,
            NUM_DAYS_TO_KEEP_OLD,
            NUM_DAYS_TO_KEEP_INF,
            "FACT_SNIP_SCORE_PRECISION_MAP",
        )


class UpdatePerSourcePqListResourceInArcadia(sdk2.Task):
    """ Update resource ID in arcadia"""

    class Parameters(sdk2.Parameters):
        resource_id = sdk2.parameters.Integer('Per-fact-source PQ list resource ID', required=True)
        arcadia_config_path = sdk2.parameters.ArcadiaUrl(
            'Path to the config with the resource ID',
            required=True,
            default_value=DEFAULT_PER_SOURCE_PQ_LIST_RESOURCE_CONFIG_PATH
        )
        author = sdk2.parameters.String('Author of commit', default='zomb-sandbox-rw', required=True)

    def on_execute(self):
        trg_line_pattern = "FROM_SANDBOX\\(FILE ({}) OUT fact_source_pq.tsv\\)"
        update_resource_in_arcadia(
            self.Parameters.resource_id,
            self.Parameters.arcadia_config_path,
            PER_SOURCE_PQ_LIST_RESOURCE_CONFIG_NAME,
            self.Parameters.author,
            trg_line_pattern,
            NUM_DAYS_TO_KEEP_OLD,
            NUM_DAYS_TO_KEEP_INF,
            "FACT_SOURCE_PQ_LIST",
        )
