import logging
import platform
import pathlib2
import os

from sandbox import sdk2
from sandbox import common
from sandbox.sdk2.helpers import subprocess
from sandbox.common.utils import get_task_link

from sandbox.projects.fastbuild.utils.git import GitAdapter
from sandbox.projects.fastbuild.utils.rvm import RvmEnvironment
from sandbox.projects.fastbuild.utils.shell import Shell
from sandbox.projects.fastbuild.utils.xcode import Xcode


class FastBuildExecute(sdk2.Task):
    """ Main task to build generic iOS projects with fastbuild extension to FastLane with Sandbox """

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600

        with sdk2.parameters.Group("Project Related") as proj_params:
            project_path = sdk2.parameters.String(
                "Project Path",
                description="project path (relative to root of repo) where to build project",
                default="/common/tools/fastbuild",
                required=True
            )
            xcode_version_required = sdk2.parameters.String(
                "Required Xcode Version",
                description="required version of Xcode to build with, should be in 8.2.1 form. Must be identical to supplied in your Fastfile. Path will be used /Application/Xcode<version>",
                default="10.1",
                required=True
            )
        with sdk2.parameters.Group("FastLane") as fastlane_params:
            fastlane_lanes = sdk2.parameters.List(
                "Lanes",
                description="list of lanes to execute by fastlane",
                default=[],
                required=True
            )
            fastlane_parameters = sdk2.parameters.Dict(
                "Parameters",
                description="dictionary of parameters to fastlane",
                required=False
            )
        with sdk2.parameters.Group("Security") as security:
            ssh_key = sdk2.parameters.Vault(
                "SSH Key",
                description="SSH key(from vault) to use to clone rtepo in git and to use to communicate with yav",
                default="",
                required=True
            )
            ssh_rsa_login = sdk2.parameters.String(
                "SSH RSA Login",
                description="Login to use for ssh agent operations",
                default="",
                required=True
            )
        with sdk2.parameters.Group("GIT") as git_params:
            git_url = sdk2.parameters.String(
                "URL",
                description="git address (with .git suffix) from which to clone the repository",
                required=True
            )
            git_branch = sdk2.parameters.String(
                "Branch",
                description="which git branch to build",
                default="master",
                required=True
            )
            git_use_latest_revision = sdk2.parameters.Bool(
                "Use Latest Revision",
                description="Should checkout submodules with --remote",
                default=False,
                required=False
            )
            git_use_lfs = sdk2.parameters.Bool(
                "Use LFS",
                description="Should use LFS",
                default=False,
                required=False
            )

    class Requirements(sdk2.Requirements):
        environments = [RvmEnvironment()]

    def on_execute(self):
        logging.info("FastBuild, building project at path %s...", self.Parameters.project_path)
        self._check_environment()
        self._clone_repository()
        self._prepare_ruby_and_related()
        self._symlink_fastbuild('fastlane')
        self._run_fastlane(self.fastlane_properties_combined)
        # temporary

    def _check_environment(self):
        logging.info("FastBuild, checking environment...")

        logging.info("Check Operating System environment...")
        if platform.system() != 'Darwin':
            logging.info("System is not 'Darwin', FastBuild can not be executed on non MacOS systems")
            raise common.errors.TaskFailure("Not a Darwin platform")
        logging.info("Darwin platform checked successfully!")

        mac_version = platform.mac_ver()
        if mac_version[0] == '':
            logging.info("Could not determine version of MacOS, it is unsupported")
            raise common.errors.TaskFailure("MacOS version is undetermined - {}".format(mac_version[0]))
        logging.info("Version of MacOS acquired - {}!".format(mac_version[0]))

        if mac_version[0] < '10.13':
            logging.info("MacOS is older than HighSierra (10.13), it is unsupported")
            raise common.errors.TaskFailure("Not supported version of MacOS - {}".format(mac_version[0]))
        logging.info("We are on supported version of MacOS - {}!".format(mac_version[0]))

        logging.info("Checking Xcode version...")
        xcode = Xcode()
        xcode_versions = xcode._get_versions()
        logging.info("Found versions of Xcode: {}".format("; ".join(xcode_versions)))
        if self.Parameters.xcode_version_required not in xcode_versions:
            logging.info("Required Xcode version {} is not installed on this agent".format(self.Parameters.xcode_version_required))
            raise common.errors.TaskFailure("No installed version {} of Xcode, required {}".format("; ".join(xcode_versions), self.Parameters.xcode_version_required))
        logging.info("Required version of Xcode {} is found!".format(self.Parameters.xcode_version_required))

    def _clone_repository(self):
        logging.info("Cloning repository for build...")
        git = GitAdapter()
        git.clone(
            self._clone_dir(),
            self.Parameters.git_url,
            self.Parameters.git_branch,
            self.Parameters.ssh_key,
            git_use_lfs=self.Parameters.git_use_lfs,
            git_use_latest_revision=self.Parameters.git_use_latest_revision
        )
        logging.info("Repository is cloned at {}.".format(self._clone_dir()))

    def _prepare_ruby_and_related(self):
        logging.info("Preparing ruby environment, ruby version and gemset for project...")
        ruby_dir = self._work_dir()
        logging.info("Ruby working directory is {}".format(ruby_dir))

        ruby_version_file = pathlib2.Path(self._work_dir(".ruby-version"))
        logging.info("Checking .ruby-version file at {}".format(ruby_version_file))
        if not ruby_version_file.exists():
            logging.info("Ruby version file at {} does not exist, exitting...".format(ruby_version_file))
            raise common.errors.TaskFailure("Not .ruby-version file supplied at {}{}!".format(self._clone_dir(), self.Parameters.project_path))
        ruby_version = ruby_version_file.read_text().strip()
        logging.info("RVM should install ruby of version {}".format(ruby_version))

        logging.info("Installing ruby version {} with RVM...".format(ruby_version))
        shell = Shell()
        ruby_install_res = shell.execute(
            ['rvm', 'install', 'ruby-' + ruby_version],
            ruby_dir, env=self.env, timeout=self.execute_shell_timeout
        )
        if ruby_install_res:
            logging.info("Can not install ruby version {} with RVM, error code: {}".format(ruby_version, ruby_install_res))
            raise common.errors.TaskFailure("Can not install ruby version {} with RVM, error code: {}".format(ruby_version, ruby_install_res))
        logging.info("Successfully installed ruby version {} with RVM at {}".format(ruby_version, ruby_version_file))

        logging.info("Cleaning gemset if any exists...")
        clean_gems_res = shell.execute(
            ['rvm', 'in', ruby_dir, 'do', 'rvm', 'gemset', 'empty', '--force'],
            ruby_dir, env=self.env, timeout=self.execute_shell_timeout
        )
        if clean_gems_res:
            logging.info("Can not clean gemset if any exists, error code: {}".format(clean_gems_res))
            raise common.errors.TaskFailure("Can not clean gemset if any exists, error code: {}".format(clean_gems_res))
        logging.info("Successfully cleanse gemset if any exists.")

        logging.info("Installing Bundler....")
        bundler_install_res = shell.execute(
            ['rvm', 'in', ruby_dir, 'do', 'gem', 'install', 'bundler', '--no-rdoc', '--no-ri'],
            ruby_dir, env=self.env, timeout=self.execute_shell_timeout
        )
        if bundler_install_res:
            logging.info("Can not install Bundler, error code: {}".format(bundler_install_res))
            raise common.errors.TaskFailure("Can not install Bundler, error code: {}".format(bundler_install_res))
        logging.info("Successfully install Bundler.")

        logging.info("Initiating project's ruby environment with Bundler...")
        bundle_initiated_res = shell.execute(
            ['rvm', 'in', ruby_dir, 'do', 'bundle', 'install', '--full-index'],
            ruby_dir, env=self.env, timeout=self.execute_shell_timeout
        )
        if bundle_initiated_res:
            logging.info("Can not initiate project's ruby environment, error code: {}".format(bundle_initiated_res))
            raise common.errors.TaskFailure("Can not initiate project's ruby environment, error code: {}".format(bundle_initiated_res))
        logging.info("Successfully initiate project's ruby environment.")

        logging.info("Ruby Environment is successfully initiated!")

    def _symlink_fastbuild(self, item):
        fromPath = self._clone_dir("common", "tools", "fastbuild", item)
        toPath = self._work_dir(item)
        if not os.path.isdir(fromPath):
            logging.info("Symlinking FastBuild's fastlane item {} from {} to {}...".format(item, fromPath, toPath))
            os.symlink(fromPath, toPath)
        else:
            if not os.path.exists(toPath):
                try:
                    os.makedirs(toPath)
                except:
                    pass
            for subitem in os.listdir(fromPath):
                # logging.info("Go deep inside {}...".format(fromPath))
                self._symlink_fastbuild((item + '/' + subitem) if subitem != '/' else '/' + subitem)

    def _run_fastlane(self, fastlane_parameters):
        def execute():
            if self.Parameters.fastlane_lanes:
                for lane in filter(None, self.Parameters.fastlane_lanes):
                    logging.info("Attempt to run fastlane lane {}".format(lane))
                    self._execute_fastlane(self._work_dir(), lane, '--verbose', fastlane_parameters)

        ssh_key = self.Parameters.ssh_key
        if ssh_key:
            with sdk2.ssh.Key(self, ssh_key.owner, ssh_key.name):
                execute()
        else:
            execute()

    def _execute_fastlane(self, path, lanes, log_level=None, properties=None, timeout=None):
        logging.info("Executing fastlane lanes {}...".format("; ".join(lanes)))
        args = ['rvm', 'in', path, 'do', 'bundle', 'exec', 'fastlane']

        if isinstance(lanes, basestring):
            args += [lanes]
        else:
            args += list(lanes)

        if properties:
            args += ["{0}:{1}".format(k, v) for (k, v) in properties.iteritems()]

        if log_level:
            args += [log_level]

        logging.info(args)
        logging.info("Fastlane arguments {}".format(" ".join(args)))
        shell = Shell()
        fastlane_res = shell.execute(args, path, env=self.env, timeout=timeout)
        if fastlane_res:
            logging.info("Can't execute fastlane lanes {}, because of error {}".format("; ".join(lanes), fastlane_res))
            raise common.errors.TaskFailure("Can't execute fastlane lanes {}, because of error {}".format("; ".join(lanes), fastlane_res))
        logging.info("Successfully execute fastlane lanes {}".format("; ".join(lanes)))

    def _clone_dir(self, *path):
        return str(self.path("work", *path))

    def _work_dir(self, *path):
        if self.Parameters.project_path:
            return str(self.path("work", self.Parameters.project_path, *path))
        else:
            return self._clone_dir(*path)

    @property
    def env(self):
        try:
            return self._env
        except AttributeError:
            self._env = dict(l.split('=', 1) for l in subprocess.check_output(['/bin/bash', '-c', '. /etc/profile && printenv']).splitlines())
            self._env['LANG'] = 'en_US.UTF-8'
        return self._env

    @property
    def execute_shell_timeout(self):
        exec_time = self.server.task.current.read()["execution"]["current"]
        remaining_time = self.Parameters.kill_timeout - exec_time
        logging.info("exec_time {0} sec, remains {1} sec, shell timeout: {1}".format(exec_time, remaining_time))
        return remaining_time

    @property
    def fastlane_properties_combined(self):
        combined = {
            "build.url": get_task_link(self.id),
            "rsa_login": self.Parameters.ssh_rsa_login
        }
        combined.update({
            k: v for k, v in self.Parameters.fastlane_parameters.iteritems() if v
        })
        # TODO: secrets
        return combined
