import logging
import tempfile

from sandbox import sdk2
from sandbox.projects.common.environments import SandboxGitLfsEnvironment
from sandbox.sdk2.vcs.git import Git


class GitAdapter:
    """
        Class to work with sandbox git utilities. Originally written by appmetrica SDK team.
    """

    def __init__(self):
        pass

    def clone(self, working_directory, git_url, git_branch, git_ssh_key, git_use_lfs=False, git_use_latest_revision=False):
        def execute(keyPath):
            env = {"GIT_CURL_VERBOSE": "1", "HOME": working_directory}
            if keyPath:
                logging.info("GIT: path to private key {}".format(keyPath))
                env["GIT_SSH_COMMAND"] = "ssh -vvv -i {}".format(keyPath)
            else:
                logging.info("GIT: path to private key is not supplied")

            git = Git(git_url)

            logging.info("GIT: Updating CACHE HEAD...")
            git.update_cache_repo('HEAD')

            logging.info("GIT: Cloning...")
            git.execute(
                'clone',
                '-vvv',
                '--reference',
                git.cache_repo_dir,
                git_url,
                working_directory,
                env=env
            )

            logging.info("GIT: Fetching...")
            git.execute(
                'fetch',
                '--verbose',
                '--tags',
                'origin',
                '+refs/heads/*:refs/heads/*',
                '+refs/pull-requests/*:refs/pull-requests/*',
                '+refs/pull/*:refs/pull/*',
                '--update-head-ok',
                cwd=working_directory,
                env=env
            )

            logging.info("GIT: Checkout branch {}".format(git_branch))
            if git_branch:
                git.execute('checkout', git_branch, cwd=working_directory, env=env)
            git.execute('branch', '-v', cwd=working_directory, env=env)

            if git_use_lfs:
                logging.info("GIT: installing git lfs hooks...")
                SandboxGitLfsEnvironment('2.7.1').prepare()
                git.execute('lfs', 'install', cwd=working_directory, env=env)
                git.execute('lfs', 'pull', cwd=working_directory, env=env)
                logging.info("GIT: lfs installed!")
            else:
                logging.info("GIT: don not use lfs")

            logging.info("GIT: Checkout submodules, ulr: {}, if applied...".format(git_use_latest_revision))
            if git_use_latest_revision:
                git.execute('submodule', 'update', '--init', '--remote', '--recursive', cwd=working_directory, env=env)
            else:
                git.execute('submodule', 'update', '--init', '--recursive', cwd=working_directory, env=env)

            logging.info("GIT: Fetching...")
            git.execute(
                'fetch',
                '--recurse-submodules=yes',
                '--tags',
                'origin',
                '+refs/heads/*:refs/heads/*',
                '+refs/pull-requests/*:refs/pull-requests/*',
                '--update-head-ok',
                cwd=working_directory,
                env=env
            )

        if git_ssh_key:
            logging.info("GIT: Using SSH key {}:{}".format(git_ssh_key.owner, git_ssh_key.name))
            k = sdk2.ssh.Key(self, git_ssh_key.owner, git_ssh_key.name)
            with k:
                logging.info("GIT: execute with SSH key support")
                with tempfile.NamedTemporaryFile() as f:
                    f.write(k.key)
                    f.flush()
                    execute(f.name)
        else:
            logging.info("GIT: execute without SSH key support")
            execute()
