import logging
import string
import time

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess


class Shell:
    """
        Class to execute shell commands. Originally written by appmetrica SDK team.
    """

    def __init__(self):
        pass

    def execute(self, args, cwd=None, env=None, timeout=None):
        """
        :params args: command-line arguments, single string or sequence
        :params cwd: current dir
        :params env: environment variables
        :return: exit code
        """
        env = env if env else self.default_environment
        with sdk2.helpers.ProcessLog(logger=self._get_logger_name(args),
                                     set_action=False,
                                     stdout_level=logging.DEBUG) as proclog:
            proclog.logger.setLevel(logging.DEBUG)
            proclog.logger.info("\t\tSHELL: command %s", args)
            proclog.logger.info("\t\tSHELL: working directory: %s", cwd)

            process = sdk2.helpers.subprocess.Popen(
                args,
                cwd=cwd,
                env=env,
                stdout=proclog.stdout,
                stderr=proclog.stderr
            )
            exit_code = process.wait(timeout=timeout)

            proclog.logger.info("\t\tSHELL: exit code %s", exit_code)

            return exit_code

    def ruby(self, args, cwd=None, env=None, timeout=None):
        env_args = ['rvm', 'in', cwd, 'do']
        return self.execute(env_args + args, cwd, env, timeout)

    def _get_logger_name(self, args):
        valid_chars = "-_.,:;() %s%s" % (string.ascii_letters, string.digits)
        logname = "{0}_{1}".format(time.time(), args[0])
        if len(args) > 1:
            logname = "{0}_{1}".format(logname, args[-1])

        return ''.join(c for c in logname if c in valid_chars)

    @property
    def default_environment(self):
        try:
            return self._env
        except AttributeError:
            self._env = dict(l.split('=', 1) for l in
                             subprocess.check_output(['/bin/bash', '-c', '. /etc/profile && printenv']).splitlines())
            self._env['LANG'] = 'en_US.UTF-8'
            return self._env
