# -*- coding: utf-8 -*-
import os
import time

import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc
from sandbox.common.types.resource import State

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.websearch.begemot.tasks.DeployBegemotFreshDataWithBstr import BegemotBstrForFreshDataDeployment, SearchReleasersLastReleasedResource
from sandbox.common.errors import TaskFailure


class FastresWizardData(sdk2.Resource):
    """
        Realtime wizard data for fast wizards (fastres)
    """
    releasable = True
    version = sdk2.Attributes.Integer("Package version", default_value=0)


class FastresWizardDataTest(sdk2.Resource):
    """
        Testing realtime wizard data for fast wizards (fastres)
    """
    releasable = True


# todo: remove support of remote file, use BuildFastresWizardData as sub task
class UploadFastresWizardData(sdk2.Task):
    """
        Downloads and releases realtime data wizard
    """

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.Group.LINUX

    class Parameters(sdk2.Task.Parameters):
        use_sandbox_resource = sdk2.parameters.Bool('Use ready sandbox resource', default_value=False)
        with use_sandbox_resource.value[True]:
            resource_id = sdk2.parameters.String('Resource id', required=True)
        with use_sandbox_resource.value[False]:
            remote_file_name = sdk2.parameters.String("Remote file name", required=True)
            remote_file_protocol = sdk2.parameters.String("Remote file protocol", required=True)
            resource_type = sdk2.parameters.String("Resource type", required=True)
            resource_name = sdk2.parameters.String("Resource name", required=True)
        skip_real_publishing = sdk2.parameters.Bool('Skip real publishing (use for test purpose)', default_value=False)
        yt_prefix = sdk2.parameters.String("yt prefix", required=True)
        bstr_resource = SearchReleasersLastReleasedResource("bstr", resource_type=BegemotBstrForFreshDataDeployment)

    def on_execute(self):
        subtasks = list(self.find().limit(0))
        if not self.Parameters.use_sandbox_resource and not subtasks:
            subtasks = []
            upload_args = {
                'remote_file_name': self.Parameters.remote_file_name,
                'remote_file_protocol': self.Parameters.remote_file_protocol,
                'created_resource_name': self.Parameters.resource_name,
                'resource_type': self.Parameters.resource_type
            }
            upload_task = sdk2.Task["REMOTE_COPY_RESOURCE"]
            subtask = upload_task(self, description="Created from UploadFastresWizardData", **upload_args).enqueue()
            subtasks.append(subtask)

            raise sdk2.WaitTask(subtasks, ctt.Status.Group.FINISH, True)

        if not self.Context.data_torrent_id:
            if not self.Parameters.bstr_resource:
                raise TaskFailure("Bstr is not available")
            self.Context.bstr_path = str(sdk2.ResourceData(self.Parameters.bstr_resource).path)

            if self.Parameters.use_sandbox_resource:
                resource = FastresWizardData.find(status=State.READY, id=self.Parameters.resource_id).first()
                if not resource:
                    raise TaskFailure("Can't find resource {}".format(self.Parameters.resource_id))
                self.Context.data_torrent_id = resource.skynet_id
            else:
                resources = list(sdk2.Resource[self.Parameters.resource_type].find(task=subtasks[0]).limit(1))
                if not resources:
                    raise TaskFailure("Upload failed: no uploaded resource found")
                self.Context.data_torrent_id = resources[0].skynet_id

            self.set_info("Skynet ID: {}".format(self.Context.data_torrent_id))
            if self.Parameters.skip_real_publishing:
                self.set_info('skip real publishing')
                return

            bstr_env = os.environ.copy()
            bstr_env['YT_TOKEN'] = sdk2.Vault.data('yt_token_for_robot-wizard-bstr')
            bstr_env['YT_LOG_LEVEL'] = 'DEBUG'

            yt_paths = self.Parameters.yt_prefix.split(',')
            total_bstr_success = True

            for p in yt_paths:
                bstr_cmd = [
                    self.Context.bstr_path,
                    'push',
                    '--root', p,
                    '--file', self.Context.data_torrent_id,
                    '-v'
                ]

                bstr_success = False
                self.set_info('Starting bstr: ' + ' '.join(bstr_cmd))
                with sdk2.helpers.ProcessLog(self, logger="bstr_pusher") as pl:
                    retries = 3
                    while retries > 0 and not bstr_success:
                        ret = sp.Popen(
                            bstr_cmd,
                            env=bstr_env,
                            stdout=pl.stdout,
                            stderr=pl.stderr
                        ).wait()
                        if ret:
                            self.set_info('Retry {}: bstr exited with nonzero code {}'.format((3 - retries), ret))
                        else:
                            bstr_success = True

                        retries = retries - 1
                        if not bstr_success:
                            time.sleep(3)
                if not bstr_success:
                    total_bstr_success = False
                    break

            if not total_bstr_success:
                raise TaskFailure("Bstr failed")
