# -*- coding: utf-8 -*-
import logging
import os
import tempfile
import time
from contextlib import contextmanager

from sandbox import sdk2
from sandbox.common.types.task import Status
from sandbox.common.types.resource import State
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.fastres.UploadFastresWizardData import FastresWizardData
from sandbox.common.errors import TaskFailure


class FastresWizardDataBuilder(sdk2.Resource):
    pass


class FastresWizardSourceData(sdk2.Resource):
    pass


# it is ok to use fixed version of gztcompiler
class FastresGztCompiler(sdk2.Resource):
    pass


# todo: move to common
@contextmanager
def _change_dir(newdir):
    prevdir = os.getcwd()
    os.chdir(os.path.expanduser(newdir))
    try:
        yield
    finally:
        os.chdir(prevdir)


class BuildFastresWizardData(sdk2.Task):
    """ Task for building fastres/fconsole wizard data """

    _RULES_GZT_PROTO = 'search/wizard/rules/proto'

    class Requirements(sdk2.Task.Requirements):
        disk_space = 50 * 1024  # Mb

    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl('Svn url for arcadia', required=True)
        source_data = sdk2.parameters.String('Source data (rbtorrent or resource id)', required=True)
        version = sdk2.parameters.Integer('Timestamp')
        keep_text_gzt = sdk2.parameters.Bool('Keep text gzt in result')
        add_debug = sdk2.parameters.Bool('Add debug info to data (rule_id, timestamp)')
        resource_filename = sdk2.parameters.String('Resource filename, does matter for bstr', required=True)

    class Context(sdk2.Task.Context):
        source_data_upload_task_id = None

    def _get_data_builder(self):
        r = FastresWizardDataBuilder.find(state=State.READY).order(-sdk2.Resource.id).limit(1).first()
        assert r
        return str(sdk2.ResourceData(r).path)

    def _get_gzt_compiler(self):
        r = FastresGztCompiler.find(state=State.READY).order(-sdk2.Resource.id).limit(1).first()
        assert r
        return str(sdk2.ResourceData(r).path)

    def _upload_source(self, rbtorrent):
        task = sdk2.Task['REMOTE_COPY_RESOURCE'](
            self,
            remote_file_name=rbtorrent,
            remote_file_protocol='skynet',
            created_resource_name='fast_wizard_data.json',
            resource_type=FastresWizardSourceData.name
        )
        return task.enqueue().id

    def _ensure_child_task_is_successful(self, task_id):
        statuses = [t.status for t in self.find(id=task_id)]
        if not all(s == Status.SUCCESS for s in statuses):
            raise TaskFailure('Child task {} is failed, status: {}'.format(task_id, ', '.join(statuses)))

    def _init_source_data(self):
        if self.Parameters.source_data.startswith('rbtorrent:'):
            with self.memoize_stage.upload_source(commit_on_entrance=False):
                self.Context.source_data_upload_task_id = self._upload_source(self.Parameters.source_data)
                self.set_info('Wait uploading source data')
                raise sdk2.WaitTask(self.Context.source_data_upload_task_id, Status.Group.FINISH | Status.Group.BREAK)
            self._ensure_child_task_is_successful(self.Context.source_data_upload_task_id)
            resource = FastresWizardSourceData.find(
                status=State.READY, task_id=self.Context.source_data_upload_task_id).first()
        else:
            resource = FastresWizardSourceData.find(status=State.READY, id=self.Parameters.source_data).first()
        assert resource
        logging.info('found resource with source data %s', resource.id)
        return str(sdk2.ResourceData(resource).path)

    def _checkout_gzt_imports(self):
        tmp_dir = tempfile.mkdtemp('arcadia')
        with _change_dir(tmp_dir):
            logging.info('clone arcadia to %s', tmp_dir)
            arcadia_dir = sdk.do_clone(self.Parameters.arcadia_url, self)
        arcadia_dir = os.path.join(tmp_dir, arcadia_dir)
        ya = os.path.join(arcadia_dir, 'ya')
        logging.info('selective checkout %s', self._RULES_GZT_PROTO)
        with sdk2.helpers.ProcessLog(self, logger='selective_checkout') as pl:
            sp.check_call([ya, 'make', '--checkout', '-j0', self._RULES_GZT_PROTO], stdout=pl.stdout,
                          stderr=sp.STDOUT, cwd=arcadia_dir)
        return arcadia_dir

    def _build_wizard_data(self, src, builder, gzt_compiler, arc_dir):
        version = self.Parameters.version if self.Parameters.version else int(time.time())
        wd_resource = FastresWizardData(self, description='realtime wizard data', path=self.Parameters.resource_filename, version=version)
        wd_data = sdk2.ResourceData(wd_resource)
        logging.info('wizard data resource %s', wd_resource.path)
        tmp_dir = tempfile.mkdtemp('build_data')
        cmd = [
            builder,
            '-f', src,
            '-r', str(wd_data.path),
            '-w', tmp_dir,
            '--arcadia_dir', arc_dir,
            '--gzt_compiler', gzt_compiler,
            '--timestamp', str(version)
        ]
        if self.Parameters.keep_text_gzt:
            cmd.append('--keep_text_gzt')
        if self.Parameters.add_debug:
            cmd.append('--add_debug')
        with sdk2.helpers.ProcessLog(self, logger='build_wizard_data') as pl:
            sp.check_call(cmd, stdout=pl.stdout, stderr=sp.STDOUT)
        wd_data.ready()

    def on_execute(self):
        source_data = self._init_source_data()
        assert source_data
        arc_dir = self._checkout_gzt_imports()
        builder = self._get_data_builder()
        gzt_compiler = self._get_gzt_compiler()
        self.set_info('build wizard data')
        self._build_wizard_data(source_data, builder, gzt_compiler, arc_dir)
