# -*- coding: utf-8 -*-

from os.path import basename, dirname

import sandbox.common.types.client as ctc

import sandbox.projects.resource_types as resource_types
from sandbox.projects.common.build import parameters as params
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.projects.resource_types.releasers import fastres2_releasers
from sandbox import sdk2
from sandbox import common


FASTRES2_BASE_DIR = 'extsearch/wizards/fastres2'


class Fastres2DaemonExecutable(sdk2.Resource):
    '''
        Исполняемый файл админки колдунщиков fastres2
        https://nanny.yandex-team.ru/ui/#/services/catalog/parallel_fastres2_wizardservice/
    '''
    releasers = fastres2_releasers
    release_subscribers = fastres2_releasers
    releasable = True
    any_arch = False
    executable = True
    auto_backup = True
    arcadia_build_path = FASTRES2_BASE_DIR + '/daemon/daemon'
    arcadia_build_name = 'daemon'


RESOURCES = {
    'daemon': {
        'description': 'http daemon',
        'type':  Fastres2DaemonExecutable,
    },
    'evlog2stats': {
        'description': 'evlog2stats',
        'type': resource_types.EVLOG2STATS_EXECUTABLE
    },
    'evlogdump': {
        'description': 'evlogdump',
        'type': resource_types.EVLOGDUMP_EXECUTABLE
    }
}


class BuildFastres2Daemon(nanny.ReleaseToNannyTask, YaMakeTask):
    """
        Build fastres2 daemon
    """

    type = 'BUILD_FASTRES_2_DAEMON'
    execution_space = 50000
    cores = 24
    client_tags = ctc.Tag.Group.LINUX

    input_parameters = [
        params.ArcadiaUrl,
        params.ArcadiaPatch,
        params.CheckoutModeParameter,
        params.CheckoutParameter,
        params.BuildSystem,
        params.BuildType,
        params.TestParameter,
    ]

    def get_build_def_flags(self):
        return '-DSANDBOX_TASK_ID={}'.format(self.id)

    def get_resources(self):
        res = {}
        for name, conf in RESOURCES.iteritems():
            t = conf['type']
            res[name] = {
                'description': conf['description'],
                'resource_path': getattr(t, 'arcadia_build_name', basename(t.arcadia_build_path)),
                'resource_type': t,
            }
        return res

    def get_arts(self):
        return [{'path': v['type'].arcadia_build_path} for v in RESOURCES.values()]

    def get_targets(self):
        return [dirname(v['type'].arcadia_build_path) for v in RESOURCES.values()] + [FASTRES2_BASE_DIR]

    def on_release(self, additional_parameters):
        is_prod = additional_parameters['release_status'] in ('stable', 'prestable')
        is_release_build = self.ctx[params.BuildType.name] == 'release'
        is_tested = self.ctx[params.TestParameter.name]

        if is_prod:
            errors = []
            if not is_release_build:
                errors += ['tried to deploy debug binaries to production!']
            if not is_tested:
                errors += ['tried to deploy not tested binaries to production!']
            if errors:
                message = 'Failed to release task: ' + ' '.join(errors)
                self.set_info(message)
                raise common.errors.ReleaseError(message)

        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        YaMakeTask.on_release(self, additional_parameters)


__Task__ = BuildFastres2Daemon
