# -*- coding: utf-8 -*-

import logging
import os
import shutil
import tarfile
import tempfile

import sandbox.common.types.task as ctt
from sandbox import common
from sandbox import sdk2


class Fastres2ViewerResource(sdk2.Resource):
    """
    .tar.gz file containing Fastres2 viewer (binary + templates)
    """
    releasers = ['ustas', 'epar']
    release_subscribers = ['ustas', 'epar']
    releasable = True


class BuildFastres2Viewer(sdk2.Task):

    VIEWER_SRC_DIR = "extsearch/wizards/fastres2/viewer"
    VIEWER_BINARY_PATH = "extsearch/wizards/fastres2/viewer/fastres2_viewer"
    VIEWER_STATIC_DIRS = (
        "static",
        "templates"
    )

    class Parameters(sdk2.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Arcadia url",
            required=True,
        )

    class Context(sdk2.Context):
        ya_make_task_id = None
        fastres2_viewer_resource_id = None

    def on_execute(self):
        # 1. Build viewer as binary in arcadia
        if not self.Context.ya_make_task_id:
            self._ya_make()

        # 2. Create resource (binary + static templates)
        resource_id = self.Context.fastres2_viewer_resource_id
        if not resource_id:
            subtasks = self.find(status=ctt.Status.Group.SUCCEED)
            if not subtasks.count:
                raise common.errors.TaskFailure("ya make failed")

            work_path = os.path.join(self.path(), tempfile.mkdtemp())
            self._copy_subtask_resources(work_path)
            self._copy_svn_files(work_path)
            resource = Fastres2ViewerResource(self, '.tar.gz containing fastres2 viewer', 'fastres2_viewer.tar.gz')
            resource_data = sdk2.ResourceData(resource)
            with tarfile.open(str(resource_data.path), 'w:gz') as tar:
                for entry in os.listdir(work_path):
                    tar.add(os.path.join(work_path, entry), arcname=entry)
            self.Context.fastres2_viewer_resource_id = str(resource.id)
            resource_data.ready()

    def _ya_make(self):
        self.Context.ya_make_task_id = self._create_ya_make_subtask()
        waited_statuses = set(common.utils.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK))
        raise sdk2.WaitTask(
            self.Context.ya_make_task_id,
            waited_statuses,
            wait_all=True
        )

    def _create_ya_make_subtask(self):
        task_id = self.server.task({
            "children": True,
            "context": {
                'targets': self.VIEWER_SRC_DIR,
                'arts': self.VIEWER_BINARY_PATH,
                'checkout_arcadia_from_url': self.Parameters.arcadia_url,
                'build_system': 'ya',
                'build_type': 'release',
                'check_return_code': True,
                'result_single_file': True,
            },
            "description": "Building Fastres2 viewer",
            "notification": [],
            "owner": self.owner,
            "priority": {
                "class": self.Parameters.priority.cls,
                "subclass": self.Parameters.priority.cls,
            },
            "type": "YA_MAKE",
        })["id"]
        self.server.batch.tasks.start.update([task_id])
        return task_id

    def _copy_subtask_resources(self, dst_path):
        resource = list(sdk2.Resource.find(task_id=self.Context.ya_make_task_id, type='ARCADIA_PROJECT').limit(1))[0]
        resource_data = sdk2.ResourceData(resource)
        logging.info('Copying resource from ' + str(resource_data.path))
        shutil.copy(str(resource_data.path), dst_path)

    def _copy_svn_files(self, dst_path):
        checkout_path = os.path.join(self.path(), tempfile.mkdtemp())
        parsed_url = sdk2.svn.Arcadia.parse_url(self.Parameters.arcadia_url)
        directory_url = sdk2.svn.Arcadia.replace(self.Parameters.arcadia_url, path=parsed_url.path + "/" + self.VIEWER_SRC_DIR)
        sdk2.svn.Arcadia.checkout(directory_url, checkout_path)

        for d in self.VIEWER_STATIC_DIRS:
            dst_dir = os.path.join(dst_path, d)
            os.mkdir(dst_dir)
            for dir_path, _, files in os.walk(os.path.join(checkout_path, d)):
                for f in files:
                    shutil.copy(os.path.join(dir_path, f), dst_dir)
