# -*- coding: utf-8 -*-

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess

from sandbox.common.errors import TaskError

from sandbox.projects.favicon import (
    FaviconAttrsDatabase,
    FaviconAttrsWadModel,
    FaviconHttpDaemon,
    FaviconMakeWadIndex,
)

DEFAULT_MAX_DATABASE_SIZE = 60 * 2**30  # bytes


class FaviconYtTaskBase(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        yt_token_vault = sdk2.parameters.String(
            'YT_TOKEN vault name',
            default='FAVICON_YT_TOKEN')

        yt_proxy = sdk2.parameters.String('YT_PROXY', default='arnold')

        build = sdk2.parameters.Task(
            'Binaries',
            task_type='BUILD_FAVICON_ROBOT',
            required=True,
        )

        table = sdk2.parameters.List('YT table path')

    def get_resource_path(self, resource_class, build_task=None):
        if build_task is None:
            build_task = self.Parameters.build
        resource = resource_class.find(task=build_task).first()
        return sdk2.ResourceData(resource)

    def get_task_env(self):
        return {
            'YT_TOKEN': sdk2.Vault.data(self.Parameters.yt_token_vault),
            'YT_PROXY': self.Parameters.yt_proxy,
        }


class BuildFaviconAttrs(FaviconYtTaskBase):
    """
        Cooks favicon database (trie of attributes) from YT table
    """

    class Parameters(FaviconYtTaskBase.Parameters):
        kill_timeout = 6 * 60 * 60  # seconds

        with sdk2.parameters.CheckGroup('Size') as size:
            size.choices = [('16', '16'), ('32', '32'), ('120', '120')]

        # Depends on production hosts RAM size
        max_database_size = sdk2.parameters.Integer(
            'MAX_DATABASE_SIZE in bytes',
            default=DEFAULT_MAX_DATABASE_SIZE,
        )

        model_build_task = sdk2.parameters.Task(
            'Id of sandbox task with wad-model',
            task_type='BUILD_FAVICON_WAD_MODEL',
            required=True,
        )

    class Requirements(sdk2.Task.Requirements):
        disk_space = DEFAULT_MAX_DATABASE_SIZE // 1024  # Mb
        ram = DEFAULT_MAX_DATABASE_SIZE // 1024  # Mb

    class Context(sdk2.Context):
        attrs_database_size_bytes = 0

    def get_resource_path(self, resource_class, build_task=None):
        if build_task is None:
            build_task = self.Parameters.build
        resource = resource_class.find(task=build_task).first()
        return sdk2.ResourceData(resource)

    def build_attrs(self, env):
        make_wad_index = self.get_resource_path(FaviconMakeWadIndex)
        model_resource = self.get_resource_path(FaviconAttrsWadModel, self.Parameters.model_build_task)
        with sdk2.helpers.ProcessLog(self, logger='build') as pl:
            cmd = list(map(str, [
                make_wad_index.path, 'MakeIndex',
                '--wad', self.attrs_trie.path,
                '--attrs-table', self.Parameters.table[0],
                '--model', model_resource.path,
                '--server-name', self.Parameters.yt_proxy,
            ]))
            subprocess.check_call(cmd, stdout=pl.stdout, stderr=subprocess.STDOUT, env=env)

        self.Context.attrs_database_size_bytes = self.attrs_trie.path.stat().st_size

    def build(self, env):
        self.build_attrs(env)

    def check(self):
        http_daemon = self.get_resource_path(FaviconHttpDaemon)
        with sdk2.helpers.ProcessLog(self, logger='check') as pl:
            cmd = map(str, [http_daemon.path, '--check', '--attrs', self.attrs_trie.path, '--use-kv-saas'])
            subprocess.check_call(cmd, stdout=pl.stdout, stderr=subprocess.STDOUT)

    def on_execute(self):
        self.attrs_trie = sdk2.ResourceData(FaviconAttrsDatabase(
            self, self.Parameters.description, 'favicon_attrs.trie'
        ))

        env = self.get_task_env()
        self.build(env)
        self.check()

    def get_database_size(self):
        return self.Context.attrs_database_size_bytes

    def on_release(self, params):
        if self.get_database_size() > self.Parameters.max_database_size:
            raise TaskError('Tries are too large to fit into memory')


class BuildFaviconWadModel(FaviconYtTaskBase):
    """
        Cooks model of wad for favicon
    """

    class Parameters(FaviconYtTaskBase.Parameters):
        kill_timeout = 6 * 60 * 60  # seconds

    class Requirements(sdk2.Task.Requirements):
        disk_space = 1024  # Mb
        ram = 4 * 1024  # Mb

    def on_execute(self):
        env = self.get_task_env()
        make_wad_index = self.get_resource_path(FaviconMakeWadIndex)

        model_resource = sdk2.ResourceData(FaviconAttrsWadModel(
            self, self.Parameters.description, 'favicon_model'
        ))

        with sdk2.helpers.ProcessLog(self, logger='build_model') as pl:
            cmd = list(map(str, [
                make_wad_index.path, 'MakeModel',
                '--attrs-table', self.Parameters.table[0],
                '--model', model_resource.path,
                '--server-name', self.Parameters.yt_proxy,
            ]))
            subprocess.check_call(cmd, stdout=pl.stdout, stderr=subprocess.STDOUT, env=env)
